import hashlib
import io
import json
import zlib
from datetime import datetime, timezone
from pathlib import Path

from PIL import Image
from psycopg2.extras import RealDictCursor

from scraper.card_types.ccv2.character import TavernCardv2
from scraper.card_types.ccv2.funcs import count_v2char_tokens, normalize_v2card_for_comparison
from scraper.database.connection import CursorFromConnectionFromPool
from scraper.database.exif import strip_exif
from scraper.database.hash import hash_bytestream
from scraper.globals import GLOBALS
from scraper.insert import write_bytes_to_hashed_file
from scraper.log import root_logger
from scraper.nyai_me.types import NyaiCharacterItem
from scraper.sort import sort_dict

_logger = root_logger.get_child('NYAIME.INSERT')


def insert_nyai_character(char_card: TavernCardv2, node: NyaiCharacterItem, png_bytes: bytes, card_raw: str, hashed_data_path: Path, extra_metadata: dict = None):
    img = strip_exif(Image.open(io.BytesIO(png_bytes))).convert('RGB')

    # Generate the data hash
    json_bytes = json.dumps(normalize_v2card_for_comparison(sort_dict(json.loads(char_card.model_dump_json())))).encode('utf-8')
    img_bytes = io.BytesIO()
    img.save(img_bytes, format='PNG')
    m = hashlib.md5()
    m.update(json_bytes)
    m.update(img_bytes.getvalue())
    card_data_hash = m.hexdigest()

    # Finish the image processing
    img.thumbnail(GLOBALS.card_image_max_dimensions, Image.LANCZOS)
    byte_stream = io.BytesIO()
    img.save(byte_stream, format='PNG', optimize=True)

    original_hash = hash_bytestream(io.BytesIO(png_bytes))
    char_card.data.extensions.char_archive.original_card_hash = original_hash
    char_card.data.extensions.char_archive.archive_data_hash = card_data_hash
    char_card.data.extensions.char_archive.source = 'nyai.me'
    char_card.data.extensions.char_archive.source_id = node.id
    assert char_card.data.extensions.char_archive.source_id
    char_card.data.extensions.char_archive.source_url = node.image_url
    assert char_card.data.extensions.char_archive.source_url

    # Always update the file in case it was missing
    png_hash = write_bytes_to_hashed_file(byte_stream)

    # Put node data into the card.
    char_card.data.tags = node.tags
    char_card.data.creator = node.author

    def_metadata = {
        'totalTokens': count_v2char_tokens(char_card),
    }
    node_metadata = {}
    if extra_metadata:
        def_metadata.update(extra_metadata)
        node_metadata.update(extra_metadata)

    # ===============================================================================
    # SQL

    with CursorFromConnectionFromPool(cursor_factory=RealDictCursor) as cursor:
        # Get the original "added" timestamp
        cursor.execute(
            'SELECT added FROM nyaime_character WHERE author = %s AND name = %s AND id = %s',
            (node.author, node.name, node.id)
        )
        added_result = cursor.fetchone()
        if added_result is not None:
            added_ts = added_result['added']
        else:
            added_ts = datetime.now(timezone.utc)

        # Update the node.
        cursor.execute(
            """
            DELETE
            FROM nyaime_character
            WHERE author = %s
              AND name = %s
              AND id = %s;
            INSERT INTO nyaime_character (author, name, id, node, metadata, added)
            VALUES (%s, %s, %s, %s, %s, %s);
            """,
            (
                node.author, node.name, node.id,
                node.author, node.name, node.id, node.model_dump_json(), json.dumps(node_metadata), added_ts
            )
        )

        # Check if we need to update the def.
        cursor.execute(
            'SELECT * FROM nyaime_character_def WHERE author = %s AND name = %s AND id = %s ORDER BY added DESC',
            (node.author, node.name, node.id)
        )
        existing_card = cursor.fetchone()
        is_new_node = should_insert = False
        if existing_card is not None:
            old_card = normalize_v2card_for_comparison(sort_dict(existing_card['definition']))
            new_card = normalize_v2card_for_comparison(sort_dict(json.loads(char_card.model_dump_json())))
            assert new_card['data'].get('extensions') is None
            assert old_card['data'].get('extensions') is None
            if old_card != new_card or existing_card['image_hash'] != png_hash:
                is_new_node = False
                should_insert = True
        else:
            is_new_node = should_insert = True

        if should_insert:
            cursor.execute(
                """
                INSERT INTO nyaime_character_def (author, name, id, definition, "raw", image_hash, original_card_hash, metadata)
                VALUES (%s, %s, %s, %s, %s, %s, %s, %s)""",
                (node.author, node.name, node.id, char_card.model_dump_json(), zlib.compress(card_raw.encode()), png_hash, original_hash, json.dumps(def_metadata))
            )
        return should_insert, is_new_node
