import concurrent.futures
import json

from pydantic import ValidationError

from scraper import http_queue
from scraper.log import root_logger
from scraper.nyai_me.types import naime_api_to_char_item

_logger = root_logger.get_child('NYAI.FETCH')


def fetch_nyai_nodes(download_threads: int, page_limit: int = None):
    response = http_queue.add('https://api.nyai.me/', method='POST', headers={
        'Referer': 'https://nyai.me/',
        'RequestType': 'PstGetTabPage',
        'IsGuest': '1',
        'SessionToken': 'null',
        'Content-Type': 'text/plain;charset=UTF-8',
        'Origin': 'https://nyai.me'
    }, post_data='{"TabID":3,"CommentAmountDisplayed":5,"PageNumber":0}')
    if not response:
        raise Exception(f'POST nyai.me failed for unknown reason. Did the proxy master fail?')

    response.raise_for_status()
    if page_limit is None:
        page_count = response.json()['PageCount']
    else:
        page_count = page_limit
    posts = []

    def fetch_item(x):
        r_2 = http_queue.add('https://api.nyai.me/', method='POST', headers={
            'Referer': 'https://nyai.me/',
            'RequestType': 'PstGetPostPage',
            'IsGuest': '1',
            'SessionToken': 'null',
            'Content-Type': 'text/plain;charset=UTF-8',
            'Origin': 'https://nyai.me'
        }, post_data='{"PostID":' + str(x['ID']) + '}')
        r_2.raise_for_status()
        return r_2.json()['Post']

    def fetch_page(page):
        characters = []
        r_1 = http_queue.add('https://api.nyai.me/', method='POST', headers={
            'Referer': 'https://nyai.me/',
            'RequestType': 'PstGetTabPage',
            'IsGuest': '1',
            'SessionToken': 'null',
            'Content-Type': 'text/plain;charset=UTF-8',
            'Origin': 'https://nyai.me'
        }, post_data='{"TabID":3,"CommentAmountDisplayed":5,"PageNumber":' + str(page) + '}')
        r_1.raise_for_status()
        items = r_1.json()['Posts']

        with concurrent.futures.ThreadPoolExecutor(max_workers=download_threads // 2) as ex2:
            f2 = [ex2.submit(fetch_item, item) for item in items]
            for f in concurrent.futures.as_completed(f2):
                characters.append(f.result())

        _logger.info(f'Page {page + 1}/{page_count}')
        return characters

    with concurrent.futures.ThreadPoolExecutor(max_workers=download_threads) as executor:
        futures = [executor.submit(fetch_page, page) for page in range(page_count)]
        for future in concurrent.futures.as_completed(futures):
            posts.extend(future.result())

    # Do it one by one so we can catch errors
    result = []
    for item in posts:
        try:
            result.append(naime_api_to_char_item(item))
        except ValidationError:
            print(item)
            raise

    return result


def _nyai_fetch_pages(headers: dict, post_data: dict, data_key: str):
    data = []
    r_1 = http_queue.add('https://api.nyai.me/', method='POST', headers=headers, post_data=json.dumps(post_data))
    r_1.raise_for_status()
    data.extend(r_1.json()[data_key])
    if len(data):
        while True:
            post_data['PageNumber'] += 1
            r_2 = http_queue.add('https://api.nyai.me/', method='POST', headers=headers, post_data=json.dumps(post_data))
            r_2.raise_for_status()
            data.extend(r_2.json()[data_key])
            if not len(r_2.json()[data_key]):
                break
    return data


def fetch_nyai_author(name: str):
    name = str(name)

    bio_r = http_queue.add('https://api.nyai.me/', method='POST', headers={
        'Referer': 'https://nyai.me/',
        'RequestType': 'PstGetUserBio',
        'IsGuest': '1',
        'SessionToken': 'null',
        'Content-Type': 'text/plain;charset=UTF-8',
        'Origin': 'https://nyai.me',
    }, post_data='{"UserName":"' + name + '"}')
    bio_r.raise_for_status()
    bio_j = bio_r.json()

    if not bio_j['Success']:
        return None, [], [], [], [], [], bio_j['Status']

    bio: str = bio_r.json()['Content']

    discussions = _nyai_fetch_pages({
        'Referer': 'https://nyai.me/',
        'RequestType': 'PstGetUserProfilePage',
        'IsGuest': '1',
        'SessionToken': 'null',
        'Content-Type': 'text/plain;charset=UTF-8',
        'Origin': 'https://nyai.me',
    }, {"UserName": name, "TabID": 0, "PageNumber": 0}, 'Posts')

    questions = _nyai_fetch_pages({
        'Referer': 'https://nyai.me/',
        'RequestType': 'PstGetUserProfilePage',
        'IsGuest': '1',
        'SessionToken': 'null',
        'Content-Type': 'text/plain;charset=UTF-8',
        'Origin': 'https://nyai.me',
    }, {"UserName": name, "TabID": 1, "PageNumber": 0}, 'Posts')

    prompts = _nyai_fetch_pages({
        'Referer': 'https://nyai.me/',
        'RequestType': 'PstGetUserProfilePage',
        'IsGuest': '1',
        'SessionToken': 'null',
        'Content-Type': 'text/plain;charset=UTF-8',
        'Origin': 'https://nyai.me',
    }, {"UserName": name, "TabID": 2, "PageNumber": 0}, 'Posts')

    requests = _nyai_fetch_pages({
        'Referer': 'https://nyai.me/',
        'RequestType': 'PstGetUserProfilePage',
        'IsGuest': '1',
        'SessionToken': 'null',
        'Content-Type': 'text/plain;charset=UTF-8',
        'Origin': 'https://nyai.me',
    }, {"UserName": name, "TabID": 4, "PageNumber": 0}, 'Posts')

    meta = _nyai_fetch_pages({
        'Referer': 'https://nyai.me/',
        'RequestType': 'PstGetUserProfilePage',
        'IsGuest': '1',
        'SessionToken': 'null',
        'Content-Type': 'text/plain;charset=UTF-8',
        'Origin': 'https://nyai.me',
    }, {"UserName": name, "TabID": 4, "PageNumber": 10}, 'Posts')

    return bio, discussions, questions, prompts, requests, meta, None
