import base64
import json
import subprocess
import traceback
from pathlib import Path
from typing import Union

from scraper.card_types.ccv2.character import TavernCardv2
from scraper.card_types.ccv3.character import CharacterCardV3
from scraper.card_types.infer import infer_anyspec_card
from scraper.log import root_logger

logger = root_logger.get_child('IMAGE')


def parse_card_metadata(image_path: Path, source: str) -> tuple[None, None, str] | tuple[str, CharacterCardV3 | TavernCardv2, str]:
    try:
        chara_data = extract_chara_data(image_path, 'chara')
        if not chara_data:
            ccv3_data = extract_chara_data(image_path, 'ccv3')
            decoded_data = ccv3_data
        else:
            decoded_data = chara_data
    except InvalidPng:
        return None, None, 'Invalid PNG'
    except Exception as e:
        return None, None, f'Failed to extract char data: "{e}"'

    if not decoded_data:
        return None, None, 'Invalid PNG metadata'

    try:
        card_parsed = infer_anyspec_card(json.loads(decoded_data), source=source)
        return decoded_data, card_parsed, ''
    except Exception as e:
        return None, None, f'Failed to parse card: {e}\n{traceback.format_exc()}'


class InvalidPng(Exception):
    pass


def extract_chara_data(image_path, key: str = 'chara'):
    cmd = f'identify -verbose "{image_path}" | grep "{key}:" | awk "{{print $2}}"'
    result = subprocess.run(cmd, shell=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    if result.returncode != 0:
        raise InvalidPng('Invalid PNG')
    chara_data = result.stdout.decode()
    decoded_data = None
    if chara_data is not None:
        chara_data = chara_data.strip(f'{key}: ').strip()
        decoded_data = base64.b64decode(chara_data).decode('utf-8', errors='ignore')
    return decoded_data
