import time
import traceback
import warnings
from io import BytesIO
from pathlib import Path
from typing import Union
from urllib.parse import urlparse

from PIL import Image
from PIL.Image import DecompressionBombWarning
from requests import Response

from scraper import http_queue
from scraper.log import root_logger
from scraper.suicide import watchdog_suicide

_logger = root_logger.get_child('CHUB.DOWNLOAD')


def download_file(url, filename: Union[Path, str] = None, ignore_404: bool = False, **kwargs) -> None | bytes:
    """
    Download a file from a given URL and save it to a specified location. Adds an item to the HTTP queue,
    and additional kwargs can be passed.

    :param url: The URL of the file to download.
    :param filename: The location to save the downloaded file.
    :param ignore_404: Whether to ignore 404 errors.
    :param kwargs: Additional arguments to pass to the request.
    """
    r = http_queue.add(url, **kwargs)
    if r is None:
        _logger.error(f'Download failed: "{url}" - no response')
        return
    if r.status_code == 200:
        if not len(r.content):
            _logger.error(f'download_file got 0 bytes for request: {url}')
        if filename:
            with open(filename, 'wb') as f:
                f.write(r.content)
            if not Path(filename).exists():
                _logger.error(f'download_file double-check failed! Exiting! {filename}')
                watchdog_suicide()
        else:
            return r.content
    else:
        handle_download_error(r, ignore_404)


def handle_download_error(response, ignore_404):
    """
    Handle errors that occur during file download.

    :param response: The response from the download request.
    :param ignore_404: Whether to ignore 404 errors.
    """
    msg = f'Download failed {response.status_code} -> {response.url}'
    if response.status_code == 404:
        if ignore_404:
            _logger.debug(msg)
        else:
            _logger.warning(msg)
    else:
        _logger.error(msg)


def download_image(url: str) -> tuple[bytes | None, str | None, Response | None]:
    parsed_url = urlparse(url)
    response = http_queue.add(url)
    if response is None:
        return None, f'Failed to download image: "{url}"', None
    if response.status_code == 404 and parsed_url.netloc in ['litterbox.catbox.moe', 'litter.catbox.moe']:
        return None, f'404: "{url}"', None
    img_bytes = response.content
    try:
        with warnings.catch_warnings():
            warnings.simplefilter('error', DecompressionBombWarning)  # shut up
            img = Image.open(BytesIO(img_bytes))
    except Exception as e:
        return None, f'{e.__class__.__name__}: {e}', None
    try:
        img.load()
        return img_bytes, None, response
    except Exception as e:
        return None, f'{e.__class__.__name__}: {e}', None


def fetch_json(url: str):
    last_traceback = None
    last_status_code = None
    for i in range(3):
        r = http_queue.add(url)
        if r is None:
            return None, None, 'no response', None
        last_status_code = r.status_code
        if r.status_code == 404 or r.status_code == 500:
            # 404 errors are usually fine because it means the card hasn't finished processing yet
            # or was deleted between the time we scraped the node list and now.
            return None, None, None, last_status_code
        elif r.status_code != 200:
            time.sleep(5)
            continue
        try:
            return r.json(), r.text, None, None
        except:
            last_traceback = traceback.format_exc()
            time.sleep(5)
            continue
    return None, None, last_traceback, last_status_code
