import os
import time
from concurrent.futures import ThreadPoolExecutor
from pathlib import Path

from scraper.chub.common_scrape import fetch_chub_api_json
from scraper.chub.types.chub_user import ChubUser, ChubUserNode
from scraper.chub.users.insert import insert_chub_user
from scraper.chub.users.scrape import scrape_chub_user_avatar
from scraper.log import root_logger

_logger = root_logger.get_child('USER.SCRAPE')
_script_path = Path(os.path.dirname(os.path.realpath(__file__)))


def process_chub_user(hash_dir: Path, username: str):
    start = time.time()

    with ThreadPoolExecutor() as executor:
        user_profile_future = executor.submit(fetch_chub_api_json, f'/api/users/{username}')
        avatar_future = executor.submit(scrape_chub_user_avatar, username)

    user_profile, user_last_traceback, user_last_status_code = user_profile_future.result()
    if user_last_traceback or user_profile is None:
        _logger.error(f'Failed to fetch user profile for "{username}" - code: {user_last_status_code} - {user_last_traceback}')
        return None, None

    avatar_png_bytes: bytes = avatar_future.result()
    if avatar_png_bytes is None:
        avatar_png_bytes = (_script_path / 'default-user-avatar.png').read_bytes()

    user_node = ChubUserNode(**user_profile)
    chub_user = ChubUser(
        user=user_node,
        png_bytes=avatar_png_bytes
    )
    insert_chub_user(chub_user, hash_dir)
    return username, time.time() - start
