import io
from datetime import datetime
from pathlib import Path

from PIL import Image

from scraper.chub.types.chub_user import ChubUser
from scraper.database.connection import CursorFromConnectionFromPool
from scraper.database.exif import strip_exif
from scraper.globals import GLOBALS
from scraper.insert import write_bytes_to_hashed_file
from scraper.log import root_logger

_logger = root_logger.get_child('CHUB.CHAR.INSERT')


def insert_chub_user(user: ChubUser, hashed_data_path: Path):
    try:
        img = strip_exif(Image.open(io.BytesIO(user.png_bytes))).convert('RGB')
    except ValueError as e:
        _logger.error(f'Failed to convert image "{user.user.username}": {e}')
        return

    img.thumbnail(GLOBALS.chub_user_image_max_dimensions, Image.LANCZOS)
    byte_stream = io.BytesIO()
    img.save(byte_stream, format='PNG', optimize=True)

    # Always update the file in case it was missing
    png_hash = write_bytes_to_hashed_file(byte_stream)

    # ===============================================================================
    # SQL

    with CursorFromConnectionFromPool() as cursor:
        # Get the original "added" timestamp
        cursor.execute(
            'SELECT added FROM chub_user WHERE username = %s AND id = %s',
            (user.user.username, user.user.id)
        )
        added_result = cursor.fetchone()
        if added_result:
            added_ts = added_result[0]
        else:
            added_ts = datetime.now()

        cursor.execute(
            """
            DELETE FROM chub_user WHERE username = %s OR id = %s;
            INSERT INTO chub_user (username, id, data, image_hash, added)
            VALUES (%s, %s, %s, %s, %s);
            """,
            (user.user.username, user.user.id, user.user.username, user.user.id, user.user.json(), png_hash, added_ts)
        )
