import json
from datetime import datetime, timezone
from json import JSONDecodeError
from typing import Any, Dict, List, Optional, Union

from pydantic import BaseModel, Field, field_validator

from scraper.card_types.item_extensions import parse_date_field


class ChubBaseNode(BaseModel):
    author: str
    lastActivityAt: datetime = datetime.fromtimestamp(0)
    createdAt: datetime = datetime.fromtimestamp(0)
    creatorId: Optional[int] = Field(None, alias="creatorId")
    description: str
    forks: List = Field(default_factory=list)
    forksCount: int
    fullPath: str
    hasGallery: bool
    id: int
    labels: List[Dict[Any, Any]] = Field(default_factory=list)
    nMessages: Optional[int] = None
    nTokens: int = -1
    n_favorites: int
    n_public_chats: Optional[int] = None
    name: str
    nsfw_image: bool = False
    primaryFormat: Union[str, None]
    projectSpace: str
    rating: float
    ratingCount: int
    related_characters: List = Field(default_factory=list)
    related_lorebooks: List = Field(default_factory=list)
    related_prompts: List = Field(default_factory=list)
    starCount: int
    tagline: str
    topics: List[str] = Field(default_factory=list)
    is_unlisted: bool = False
    avatar_url: Optional[str] = None
    bound_preset: Optional[str] = None
    verified: Optional[bool] = False
    recommended: Optional[bool] = False
    ratings_disabled: Optional[bool] = False

    class Config:
        json_encoders = {
            datetime: lambda v: v.astimezone(timezone.utc).isoformat()
        }
        validate_assignment = True

    @field_validator('createdAt', 'lastActivityAt', mode='before')
    @classmethod
    def parse_date(cls, v):
        if v:
            return parse_date_field(v)
        else:
            return datetime.fromtimestamp(0)


def clean_chub_node(node_data: dict) -> dict:
    if node_data.get('is_favorite'):
        del node_data['is_favorite']
    if node_data.get('is_public'):
        del node_data['is_public']
    if node_data.get('permissions'):
        del node_data['permissions']
    if node_data.get('rating'):
        node_data['rating'] = float(node_data['rating'])
    if not node_data.get('related_characters'):
        node_data['related_characters'] = []
    if not node_data.get('related_lorebooks'):
        node_data['related_lorebooks'] = []
    if not node_data.get('related_prompts'):
        node_data['related_prompts'] = []
    if not node_data.get('nsfw_image'):
        node_data['nsfw_image'] = False
    if not node_data.get('expressions'):
        node_data['expressions'] = {}
    if isinstance(node_data.get('n_public_chats', 0), bool):
        node_data['n_public_chats'] = 0

    if not node_data.get('author') and node_data.get('fullPath'):
        full_path = node_data['fullPath'].split('/')
        if full_path[0] == 'lorebooks':
            full_path.pop(0)
        node_data['author'] = full_path[0]

    # We don't care about this.
    node_data['definition'] = {}

    # Covert the things in "labels" from JSON to a dict.
    for i, item in enumerate(node_data.get('labels', [])):
        for k, v in item.items():
            if isinstance(v, str):
                try:
                    node_data['labels'][i][k] = json.loads(v)
                except JSONDecodeError:
                    pass

    return node_data
