import warnings
from typing import List, Dict, Any

from pydantic import BaseModel, Field

from scraper.card_types.ccv2.character import TavernCardv2
from scraper.chub.common_scrape import normalize_for_chub
from scraper.chub.types.chats_ratings import ChubChats, ChubRatings
from scraper.chub.types.node import ChubBaseNode

warnings.simplefilter('error', UserWarning)


class ChubCharExpressions(BaseModel):
    version: str = None
    compressed: str = None
    expressions: Dict[str, Any] = Field(default_factory=dict)


class ChubCharNode(ChubBaseNode):
    related_extensions: List = Field(default_factory=list)
    nChats: int = 0
    expressions: ChubCharExpressions = ChubCharExpressions()


class ChubCharacter:
    def __init__(self, node: ChubCharNode, card_data: TavernCardv2, png_bytes: bytes, chats: ChubChats = None, ratings: ChubRatings = None):
        assert isinstance(card_data, TavernCardv2)
        self._node = node
        self._card_data = normalize_for_chub(card_data, node.id, node.fullPath)
        self._png_bytes = png_bytes

        if chats:
            self._chats = chats
        else:
            self._chats = ChubChats()

        if ratings:
            self._ratings = ratings
        else:
            self._ratings = ChubRatings()

    @property
    def node(self) -> ChubCharNode:
        return self._node

    @property
    def card_data(self) -> TavernCardv2:
        return self._card_data

    @property
    def png_bytes(self) -> bytes:
        return self._png_bytes

    @property
    def chats(self) -> ChubChats:
        return self._chats

    @property
    def ratings(self) -> ChubRatings:
        return self._ratings
