from datetime import datetime, timezone
from typing import List, Dict, Any, Optional

from pydantic import BaseModel, Field, field_validator

from scraper.card_types.item_extensions import parse_date_field


class ChubChats(BaseModel):
    chats: List[Dict[Any, Any]] = Field(default_factory=list)


class ChubRatingItem(BaseModel):
    id: Optional[int] = None
    project_id: int = Field(alias='projectId')
    rating: Optional[int] = None
    created_at: Optional[datetime] = Field(alias='createdAt')
    updated_at: Optional[datetime] = Field(alias='updatedAt')
    comment: Optional[str] = None
    comment_at: Optional[datetime] = Field(alias='commentAt')
    comment_updated_at: Optional[datetime] = Field(alias='commentUpdatedAt')
    parent_id: Optional[int] = Field(None, alias='parentId')
    children: List[int] = Field(default_factory=list)
    is_mine: bool = Field(alias='isMine')
    username: Optional[str] = None
    avatar_url: Optional[str] = Field(None, alias='avatarUrl')
    badge: Optional[str] = None

    class Config:
        json_encoders = {
            datetime: lambda v: v.astimezone(timezone.utc).isoformat()
        }
        validate_assignment = True
        populate_by_name = True  # This allows both snake_case and camelCase field names

    @field_validator('comment_at', 'created_at', 'updated_at', 'comment_updated_at', mode='before')
    @classmethod
    def parse_date(cls, v):
        if v:
            return parse_date_field(v)
        else:
            return None  # Changed from datetime.fromtimestamp(0) to None for optional fields


class ChubRatings(BaseModel):
    enabled: Optional[bool] = True
    ratings: List[ChubRatingItem] = Field(default_factory=list)
    ratings_map: Dict[int, Any] = Field(default_factory=dict)
    parents: List[int] = Field(default_factory=list)

# def normalize_chub_ratings(chub_comments: list):
#     result = []
#     for comment in chub_comments:
#         if comment.get('commentAt'):
#             comment['comment_at'] = comment.pop('commentAt')
#         if comment.get('commentUpdatedAt'):
#             comment['comment_updated_at'] = comment.pop('commentUpdatedAt')
#         if comment.get('createdAt'):
#             comment['created_at'] = comment.pop('createdAt')
#         if comment.get('parentId'):
#             comment['parent_id'] = comment.pop('parentId')
#         if comment.get('projectId'):
#             comment['project_id'] = comment.pop('projectId')
#         if comment.get('updatedAt'):
#             comment['updated_at'] = comment.pop('updatedAt')
#         if comment.get('userId'):
#             comment['user_id'] = comment.pop('userId')
#
#         result.append(ChubRatingItem(**comment))
#     return result
