import functools
from concurrent.futures import ThreadPoolExecutor, as_completed
from datetime import datetime
from pathlib import Path

import numpy as np

from scraper.log import root_logger
from scraper.stop_event import global_stop_event
from scraper.suicide import watchdog_suicide
from scraper.time import calculate_elapsed_time

logger = root_logger.get_child('CHUB.RUNNER')


def run_chub_scrape(scrape_type: str, nodes: list, function, threads: int):
    start = datetime.now()
    processed_count = 0
    new_items = 0
    updated_items = 0
    execution_duration = []
    found_users = set()

    with (ThreadPoolExecutor(max_workers=threads) as executor):
        futures = {executor.submit(functools.partial(function), node) for node in nodes}
        for future in as_completed(futures):
            result = future.result()
            if global_stop_event.is_set():
                return

            node, card, elapsed, is_updated, is_new, author = result
            if not node and not card:
                # The downloader skipped this item
                continue

            found_users.add(author)

            mod_str = ''
            if is_updated:
                mod_str = ' (modified)'
            elif is_new:
                mod_str = ' (new)'
            logger.info(f'{scrape_type} completed in {int(elapsed)}s: "{node.fullPath}"{mod_str}')

            processed_count += 1
            if is_updated:
                updated_items += 1
            elif is_new:
                new_items += 1
            if is_updated and is_new:
                logger.critical(f'Card {node.author}/{node.name} was marked as new and updated')
                watchdog_suicide()

            # TODO: track new and updated nodes

            execution_duration.append(elapsed)

    time_str = calculate_elapsed_time(start)
    logger.info(f'-- {scrape_type}s scraping completed in {time_str} --'.upper())

    if len(execution_duration):
        avg_execution_duration = np.average(execution_duration)
    else:
        avg_execution_duration = 0
    return processed_count, new_items, updated_items, avg_execution_duration, found_users
