import os
import time
import traceback
from concurrent.futures import ThreadPoolExecutor
from pathlib import Path

from pydantic import ValidationError

from scraper.card_types.ccv2.lorebook import normalize_lorebook
from scraper.card_types.funcs import test_card_name
from scraper.chub.chub import CHUB_TYPE_STR
from scraper.chub.common_scrape import scrape_chub_ratings
from scraper.chub.download import get_chub_node_avatar
from scraper.chub.get import get_chub_node_def
from scraper.chub.lorebooks.insert import insert_chub_lorebook
from scraper.chub.strings import split_full_path
from scraper.chub.types.chats_ratings import ChubRatings
from scraper.chub.types.chub_lorebook import ChubLorebook, normalize_chub_lore_node
from scraper.log import root_logger

_logger = root_logger.get_child('CHUB.LORE.PROCESS')
_script_path = Path(os.path.dirname(os.path.realpath(__file__)))


def process_lore_node(node: dict):
    start = time.time()
    try:
        author, card_path = split_full_path(node['fullPath'])
        node['author'] = author
        lore_node = normalize_chub_lore_node(node)
    except (ValidationError, ValueError) as e:
        _logger.error(f'Node "{node.get("fullPath", "None")}" failed validation: {e}')
        return None, None, None, None, None, None
    del node  # avoid any issues with calling the wrong var

    # If this card was updated after since, skip the card.
    # if since and lore_node.lastActivityAt < since:
    #     return None, None, None, None, None, None

    with ThreadPoolExecutor() as executor:
        # Fetch the definition.
        def_future = executor.submit(get_chub_node_def, lore_node, 'raw%252Fsillytavern_raw.json', )

        # Download the avatar PNG.
        avatar_future = executor.submit(get_chub_node_avatar, lore_node.fullPath)

        # Download ratings.
        ratings_future = executor.submit(scrape_chub_ratings, lore_node.id)

        # =======================================================================================
        # WAIT

        lore_json, lore_raw, def_err, def_status_code = def_future.result()
        if def_err or lore_json is None:
            _logger.error(f'Failed to fetch lorebook def for "{lore_node.fullPath}" - code: {def_status_code} - {def_err}')
            return None, None, None, None, None, None
        lorebook_obj = normalize_lorebook(lore_json, CHUB_TYPE_STR)

        if not test_card_name(lorebook_obj.name):
            if not test_card_name(lore_node.name):
                _logger.debug(f'Empty lorebook name: "{lore_node.fullPath}"')
                return None, None, None, None, None, None
            else:
                lorebook_obj.name = lore_node.name

        avatar_png_bytes: bytes = avatar_future.result()
        if not avatar_png_bytes:
            avatar_png_bytes = (_script_path / 'default-lorebook-avatar.png').read_bytes()

        try:
            ratings_d, ratings_trace, ratings_status_code = ratings_future.result()
            if ratings_trace or ratings_d is None:
                _logger.warning(f'Failed to fetch ratings for "{lore_node.fullPath}" - code: {ratings_status_code} - {ratings_trace}')
                ratings = ChubRatings()
            else:
                ratings = ChubRatings.model_validate(ratings_d)
        except:
            _logger.warning(f'Failed to fetch ratings for "{lore_node.fullPath}" - {traceback.format_exc()}')
            ratings = ChubRatings()

        chub_lorebook = ChubLorebook(
            node=lore_node,
            lorebook=lorebook_obj,
            png_bytes=avatar_png_bytes,
            ratings=ratings
        )
        is_updated, is_new = insert_chub_lorebook(chub_lorebook, lore_raw)
        return lore_node, chub_lorebook, time.time() - start, is_updated, is_new, chub_lorebook.node.author
