import io
import json
import zlib
from datetime import datetime
from pathlib import Path

from PIL import Image
from psycopg2.extras import RealDictCursor

from scraper.card_types.ccv2.lorebook import count_v2lorebook_tokens
from scraper.chub.types.chub_lorebook import ChubLorebook
from scraper.database.connection import CursorFromConnectionFromPool
from scraper.database.exif import strip_exif
from scraper.database.hash import hash_bytestream
from scraper.globals import GLOBALS
from scraper.insert import write_bytes_to_hashed_file
from scraper.log import root_logger
from scraper.sort import sort_dict

_logger = root_logger.get_child('CHUB.LORE.INSERT')


def normalize_lorebook_for_comparison(book: dict):
    if book['extensions'].get('char_archive'):
        del book['extensions']['char_archive']
    return book


def insert_chub_lorebook(lorebook: ChubLorebook, lorebook_raw: str):
    assert lorebook.node.author != 'lorebooks'

    try:
        img = strip_exif(Image.open(io.BytesIO(lorebook.png_bytes))).convert('RGB')
    except ValueError as e:
        _logger.error(f'Failed to convert image "{lorebook.node.fullPath}": {e}')
        return False, False

    img.thumbnail(GLOBALS.card_image_max_dimensions, Image.LANCZOS)
    byte_stream = io.BytesIO()
    img.save(byte_stream, format='PNG', optimize=True)

    # Always update the file in case it was missing
    png_hash = write_bytes_to_hashed_file(byte_stream)

    lorebook.lorebook.name = lorebook.node.name
    original_hash = hash_bytestream(io.BytesIO(lorebook_raw.encode()))
    lorebook.lorebook.extensions.char_archive.original_card_hash = original_hash
    if lorebook.node.createdAt != datetime.fromtimestamp(0):
        lorebook.lorebook.extensions.char_archive.created = lorebook.node.createdAt
        lorebook.lorebook.create_date = lorebook.node.createdAt

    metadata = {
        'totalTokens': count_v2lorebook_tokens(lorebook.lorebook)
    }

    # ===============================================================================
    # SQL

    with CursorFromConnectionFromPool(cursor_factory=RealDictCursor) as cursor:
        # Get the original "added" timestamp
        cursor.execute(
            'SELECT added FROM chub_lorebook WHERE author = %s AND name = %s AND id = %s',
            (lorebook.node.author, lorebook.node.name, lorebook.node.id)
        )
        added_result = cursor.fetchone()
        if added_result:
            added_ts = added_result['added']
        else:
            added_ts = datetime.now()

        cursor.execute(
            """
            DELETE FROM chub_lorebook WHERE id = %s;
            INSERT INTO chub_lorebook (author, name, id, data, ratings, added)
            VALUES (%s, %s, %s, %s, %s, %s);
            """,
            (lorebook.node.id, lorebook.node.author, lorebook.node.name, lorebook.node.id, lorebook.node.json(), lorebook.ratings.json(), added_ts)
        )

        cursor.execute(
            'SELECT * FROM chub_lorebook_def WHERE author = %s AND name = %s AND id = %s AND full_path = %s ORDER BY added DESC LIMIT 1',
            (lorebook.node.author, lorebook.node.name, lorebook.node.id, lorebook.node.fullPath)
        )
        should_insert = False
        existing_cards = 0
        for item in cursor.fetchall():
            existing_cards += 1
            old_card = normalize_lorebook_for_comparison(sort_dict(item['definition']))
            new_card = normalize_lorebook_for_comparison(sort_dict(json.loads(lorebook.lorebook.json())))
            if old_card != new_card:
                should_insert = True
                break

        if should_insert or existing_cards == 0:
            cursor.execute(
                """
                INSERT INTO chub_lorebook_def (author, name, id, full_path, definition, raw, image_hash, metadata)
                VALUES (%s, %s, %s, %s, %s, %s, %s, %s);
                """,
                (lorebook.node.author, lorebook.node.name, lorebook.node.id, lorebook.node.fullPath, lorebook.lorebook.json(), zlib.compress(lorebook_raw.encode()), png_hash, json.dumps(metadata))
            )
        return should_insert, existing_cards == 0
