import random
import time
import traceback
from concurrent import futures

from ..download import download_file
from ..globals import GLOBALS
from ..http_queue import http_queue
from ..log import root_logger

logger = root_logger.get_child('CHUB.DOWNLOAD')


def download_chats(full_path):
    def dl(chat):
        r = None
        last_traceback = None
        for i in range(3):
            url = f'https://gateway.chub.ai/api/core/chats/v2/{chat["id"]}?nocache=0.{random.randrange(1, 10 ** 16):03}&include_messages=true&include_config=true&include_meta=true'
            try:
                r = http_queue.add(url)
                r.raise_for_status()
                return r.json(), None
            except:
                last_traceback = traceback.format_exc()
                time.sleep(5)
        if not r:
            return None, last_traceback

    chats_data = None
    last_traceback_root = None
    for i in range(3):
        try:
            r = http_queue.add(f'https://gateway.chub.ai/api/core/chats/{full_path}/public?nocache=0.{random.randrange(1, 10 ** 16):03}')
            if not r:
                continue
            chats_data = r.json()
        except:
            last_traceback_root = traceback.format_exc()

    if chats_data is None:
        logger.error(f'Failed to download chat for "{full_path}" -  {last_traceback_root}')
        return

    if len(chats_data['chats']):
        with futures.ThreadPoolExecutor(max_workers=3) as executor:
            for result, trace in executor.map(dl, chats_data['chats']):
                if not result:
                    logger.warning(f'Failed to get chat for "{full_path}": {trace}')
                    continue
                for i in range(len(chats_data['chats'])):
                    if chats_data['chats'][i]['id'] == result['chat']['id']:
                        chats_data['chats'][i]['node'] = result
    return chats_data


def get_chub_node_avatar(full_path: str) -> bytes | None:
    if not full_path.startswith('lorebooks'):
        png_url = f'https://avatars.charhub.io/avatars/{full_path}/chara_card_v2.png?nocache=' + str(random.randrange(1, 10 ** 16))
        png_bytes = download_file(png_url, ignore_404=True, timeout=GLOBALS.request_timeout_longer)
        if png_bytes:
            return png_bytes
    # Lorebooks are always WEBP. For cards, sometimes the image will be a WEBP instead of a PNG.
    webp_url = f'https://avatars.charhub.io/avatars/{full_path}/avatar.webp?nocache=' + str(random.randrange(1, 10 ** 16))
    webp_bytes = download_file(webp_url, ignore_404=True, timeout=GLOBALS.request_timeout_longer)
    return webp_bytes
