import time
import traceback
from concurrent.futures import ThreadPoolExecutor

from pydantic import ValidationError

from scraper.card_types.ccv2.infer import infer_specv2_card
from scraper.card_types.funcs import test_card_name
from scraper.chub.chars.insert import insert_chub_character
from scraper.chub.chars.scrape import scrape_expressions
from scraper.chub.chub import CHUB_TYPE_STR
from scraper.chub.common_scrape import scrape_chub_ratings
from scraper.chub.download import download_chats, get_chub_node_avatar
from scraper.chub.get import get_chub_node_def
from scraper.chub.strings import split_full_path
from scraper.chub.types.chats_ratings import ChubChats, ChubRatings
from scraper.chub.types.chub_character import ChubCharacter, ChubCharExpressions, ChubCharNode
from scraper.chub.types.node import clean_chub_node
from scraper.log import root_logger

_logger = root_logger.get_child('CHARS.PROCESS')


def process_char_node(node: dict):
    start = time.time()
    try:
        author, card_path = split_full_path(node['fullPath'])
        node['author'] = author
        char_node = ChubCharNode(**clean_chub_node(node))
    except (ValidationError, ValueError) as e:
        _logger.error(f'Node "{node.get("fullPath", "None")}" failed validation: {e}')
        return None, None, None, None, None, None
    del node  # avoid any issues with calling the wrong var

    # If this card was updated after since, skip the card.
    # This is commented out since the fetch process will filter the nodes and
    # will give a little buffer that is good to process.
    # if since and char_node.lastActivityAt < since:
    #     return None, None, None, None, None, None

    with ThreadPoolExecutor() as executor:
        # Download the character card as a JSON file.
        def_future = executor.submit(get_chub_node_def, char_node, 'card.json')

        # Download the card avatar PNG.
        card_avatar_png_future = executor.submit(get_chub_node_avatar, char_node.fullPath)

        # Download ratings.
        ratings_future = executor.submit(scrape_chub_ratings, char_node.id)

        # Download the card's chats.
        chats_future = executor.submit(download_chats, char_node.fullPath)

        # Download the card's expressions, if present.
        expressions_future = executor.submit(scrape_expressions, char_node)

        # =======================================================================================
        # WAIT

        card_json, card_raw, err, def_status_code = def_future.result()
        if err or card_json is None:
            _logger.error(f'Failed to fetch character def for "{char_node.fullPath}" - code: {def_status_code} - {err}')
            return None, None, None, None, None, None

        clean_node = clean_chub_node(card_json)
        if clean_node.get('data'):
            card_name = clean_node['data']['name']
        else:
            card_name = clean_node['name']
        if not test_card_name(card_name):
            if not test_card_name(char_node.name):
                _logger.debug(f'Empty character name: "{char_node.fullPath}"')
                return None, None, None, None, None, None
            else:
                if clean_node.get('data'):
                    clean_node['data']['name'] = char_node.name
                else:
                    clean_node['name'] = char_node.name

        char_card = infer_specv2_card(
            raw_card=clean_node,
            specific_author=char_node.author,
            source=CHUB_TYPE_STR
        )

        card_avatar_png_bytes: bytes = card_avatar_png_future.result()
        if not card_avatar_png_bytes:
            _logger.error(f'No avatar for card "{char_node.fullPath}"')
            return None, None, None, None, None, None

        try:
            chats_d: dict = chats_future.result()
            if not chats_d:
                chats_d = {"chats": []}
            chats = ChubChats.model_validate(chats_d)
        except:
            _logger.warning(f'Failed to fetch chats for "{char_node.fullPath}" - {traceback.format_exc()}')
            chats = ChubChats()

        try:
            ratings_d, ratings_trace, ratings_status_code = ratings_future.result()
            if ratings_trace or not ratings_d:
                _logger.warning(f'Failed to fetch ratings for "{char_node.fullPath}" - code: {ratings_status_code} - {ratings_trace}')
                ratings = ChubRatings()
            else:
                ratings = ChubRatings.model_validate(ratings_d)
        except:
            _logger.warning(f'Exception processing ratings for "{char_node.fullPath}" - {traceback.format_exc()}')
            ratings = ChubRatings()

        # We aren't archiving expressions, but want to fetch their definitions though.
        try:
            expressions, exp_trace, exp_status_code = expressions_future.result()
            if exp_trace:
                _logger.warning(f'Failed to fetch expressions for "{char_node.fullPath}" - code: {exp_status_code} - {exp_trace}')
                expressions = ChubCharExpressions()
            if not expressions:
                expressions = ChubCharExpressions()
        except Exception as e:
            _logger.warning(f'Failed to fetch expressions for "{char_node.fullPath}" - {e}')
            expressions = ChubCharExpressions()
        char_node.expressions = expressions

    char = ChubCharacter(
        node=char_node,
        card_data=char_card,
        png_bytes=card_avatar_png_bytes,
        chats=chats,
        ratings=ratings
    )
    is_updated, is_new = insert_chub_character(char, card_raw)
    return char_node, char_card, time.time() - start, is_updated, is_new, char.node.author
