import io
import json
import zlib
from datetime import datetime

from PIL import Image
from psycopg2.extras import RealDictCursor

from scraper.card_types.ccv2.funcs import count_v2char_tokens, normalize_v2card_for_comparison
from scraper.chub.strings import check_censored_image
from scraper.chub.types.chub_character import ChubCharacter
from scraper.database.connection import CursorFromConnectionFromPool
from scraper.database.exif import strip_exif
from scraper.database.hash import hash_bytestream
from scraper.globals import GLOBALS
from scraper.insert import write_bytes_to_hashed_file
from scraper.log import root_logger
from scraper.sort import sort_dict

_logger = root_logger.get_child('CHUB.CHAR.INSERT')


def insert_chub_character(char: ChubCharacter, card_raw: str, extra_metadata: dict = None):
    try:
        img = strip_exif(Image.open(io.BytesIO(char.png_bytes))).convert('RGB')
    except ValueError as e:
        _logger.error(f'Failed to convert image "{char.node.fullPath}": {e}')
        return False, False

    # Card image is always updated.
    img.thumbnail(GLOBALS.card_image_max_dimensions, Image.LANCZOS)
    byte_stream = io.BytesIO()
    img.save(byte_stream, format='PNG', optimize=True)
    png_hash = write_bytes_to_hashed_file(byte_stream)

    if check_censored_image(png_hash):
        # Shouldn't ever encounter this because the chara data shouldn't be present in the original PNG.
        _logger.error(f'Character is censored: "{char.node.fullPath}"')
        return False, False

    original_hash = hash_bytestream(io.BytesIO(char.png_bytes))
    char.card_data.data.extensions.char_archive.original_card_hash = original_hash
    if char.node.createdAt != datetime.fromtimestamp(0):
        char.card_data.data.extensions.char_archive.created = char.node.createdAt
        char.card_data.create_date = char.node.createdAt

    # Require the `chub` extension and nessesary fields.
    assert char.card_data.data.extensions.model_dump().get('chub') is not None
    assert char.card_data.data.extensions.model_dump()['chub'].get('id') is not None
    assert char.card_data.data.extensions.model_dump()['chub'].get('full_path') is not None

    metadata = {
        'totalTokens': count_v2char_tokens(char.card_data)
    }
    node_metadata = {}
    if extra_metadata:
        metadata.update(extra_metadata)
        node_metadata.update(extra_metadata)

    # ===============================================================================
    # SQL

    with CursorFromConnectionFromPool(cursor_factory=RealDictCursor) as cursor:
        # Get the original "added" timestamp
        cursor.execute(
            'SELECT added FROM chub_character WHERE author = %s AND name = %s AND id = %s',
            (char.node.author, char.node.name, char.node.id)
        )
        added_result = cursor.fetchone()

        if added_result:
            added_ts = added_result['added']
        else:
            added_ts = datetime.now()

        cursor.execute(
            """
            DELETE
            FROM chub_character
            WHERE id = %s;
            INSERT INTO chub_character (author, name, id, data, chats, ratings, added, metadata)
            VALUES (%s, %s, %s, %s, %s, %s, %s, %s);
            """,
            (char.node.id, char.node.author, char.node.name, char.node.id, char.node.model_dump_json(), char.chats.model_dump_json(), char.ratings.model_dump_json(), added_ts, json.dumps(node_metadata))
        )

        cursor.execute(
            'SELECT * FROM chub_character_def WHERE author = %s AND name = %s AND id = %s AND full_path = %s ORDER BY added DESC LIMIT 1',
            (char.node.author, char.node.name, char.node.id, char.node.fullPath)
        )
        should_insert = False
        existing_cards = 0
        for item in cursor.fetchall():
            existing_cards += 1
            old_card = normalize_v2card_for_comparison(sort_dict(item['definition']))
            new_card = normalize_v2card_for_comparison(sort_dict(json.loads(char.card_data.model_dump_json())))
            if old_card != new_card or item['image_hash'] != png_hash:
                should_insert = True
                break

        if should_insert or existing_cards == 0:
            cursor.execute(
                """
                INSERT INTO chub_character_def (author, name, id, full_path, definition, raw, image_hash, original_card_hash, metadata)
                VALUES (%s, %s, %s, %s, %s, %s, %s, %s, %s);
                """,
                (char.node.author, char.node.name, char.node.id, char.node.fullPath, char.card_data.model_dump_json(), zlib.compress(card_raw.encode()), png_hash, original_hash, json.dumps(metadata))
            )
        return should_insert, existing_cards == 0
