import time

import requests

from scraper.log import root_logger

_logger = root_logger.get_child('CHUB.AUTH')


def chub_login(username: str, password: str) -> str | None:
    """
    Logs into chub.ai and retrieves account information.

    Args:
        username (str): The user's email or username.
        password (str): The user's password.

    Returns:
        dict: A dictionary containing account information if successful.
        None: If the login fails.
    """
    session = requests.Session()

    # Common headers for all requests
    common_headers = {
        'accept-language': 'en-US,en;q=0.9',
        'access-control-allow-credentials': 'True',
        'access-control-allow-origin': '*',
        'cache-control': 'no-cache',
        'origin': 'https://chub.ai',
        'pragma': 'no-cache',
        'priority': 'u=1, i',
        'referer': 'https://chub.ai/',
        # 'samwise': 'glpat-UZXEBupEVv2vMCdFDkfJ', # Not sending this as it's set by the clientside JS?
        'sec-ch-ua': '"Not)A;Brand";v="8", "Chromium";v="138", "Google Chrome";v="138"',
        'sec-ch-ua-mobile': '?0',
        'sec-ch-ua-platform': '"Linux"',
        'sec-fetch-dest': 'empty',
        'sec-fetch-mode': 'cors',
        'sec-fetch-site': 'same-site',
        'user-agent': 'Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/138.0.0.0 Safari/537.36',
    }

    # Step 1: Get CSRF Token
    token_url = 'https://gateway.chub.ai/authentication/token'
    token_headers = common_headers.copy()

    try:
        token_response = session.get(token_url, headers=token_headers, timeout=10, proxies={'https': 'http://172.0.3.109:9000'})
        # print(token_response.text)
        token_response.raise_for_status()
        csrf_data = token_response.json()
        csrf_token = csrf_data.get('csrf_token')
        if not csrf_token:
            _logger.error('CSRF token not found in the response.')
            return None
    except requests.RequestException as e:
        _logger.error(f'Error fetching CSRF token: {e}')
        return None

    # Step 2: Perform Login
    login_url = 'https://gateway.chub.ai/authentication/login'
    login_headers = common_headers.copy()

    login_payload = {
        "csrf_token": csrf_token,
        "email_or_username": username,
        "password": password,
        "oauth": None,
        "state": "",
        "redirect_url": "https://chub.ai/login",
        "is_mobile": "false"
    }

    try:
        login_response = session.post(login_url, headers=login_headers, json=login_payload, timeout=10)
        login_response.raise_for_status()
        login_data = login_response.json()

        samwise_token = login_data.get('samwise')
        if not samwise_token:
            _logger.error('Samwise token not found in the login response.')
            return None
    except requests.RequestException as e:
        _logger.error(f'Error during login: {e}')
        return None
    except ValueError:
        _logger.error('Invalid JSON response during login.')
        return None

    # Update session headers with the new Samwise token
    session.headers.update({
        'samwise': samwise_token,
        'CH-API-KEY': samwise_token
    })

    # Step 3: Retrieve Account Information
    account_url = 'https://gateway.chub.ai/api/account'
    account_params = {
        'nocache': str(time.time())
    }
    account_headers = common_headers.copy()
    account_headers.update({
        'samwise': samwise_token,
        'CH-API-KEY': samwise_token
    })

    try:
        account_response = session.get(account_url, headers=account_headers, params=account_params, timeout=10)
        account_response.raise_for_status()
        account_data = account_response.json()

        if account_data.get('id') == -1:
            _logger.error('Invalid account ID received.')
            return None
        return samwise_token
    except requests.RequestException as e:
        _logger.error(f'Error fetching account information: {e}')
        return None
    except ValueError:
        _logger.error('Invalid JSON response when fetching account information.')
        return None
