import concurrent.futures
import io
import os
import re
import traceback
from pathlib import Path
from typing import Union, List, Set, Tuple

from PIL import Image

from scraper import http_queue
from scraper.database.connection import CursorFromConnectionFromPool
from scraper.database.exif import strip_exif
from scraper.download import download_file
from scraper.globals import GLOBALS
from scraper.insert import write_bytes_to_hashed_file
from scraper.log import root_logger

_logger = root_logger.get_child('CHAR-TAVERN.USERS')

_DEFAULT_PROFILE_PIC_PATH = Path(os.path.dirname(os.path.realpath(__file__)), 'CT_Default.png').absolute()
assert _DEFAULT_PROFILE_PIC_PATH.exists()


def _extract_user_info_from_sveltekit(html_page: str):
    match = re.search(r',views:(\d*),downloads:(\d*),displayName:".*?",bio:"?(.*?)"?,', html_page)
    if not match:
        return None, None, None, None
    else:
        views = match.group(1)
        downloads = match.group(2)
        bio = match.group(3)
        if bio == 'null':
            bio = None
        profile_img_match = re.search(r'avatarURL:"(.*?)",', html_page)
        profile_img_url = None
        if profile_img_match:
            profile_img_url = profile_img_match.group(1)
        return views, downloads, bio, profile_img_url


def _process_char_tavern_user(username: str) -> Tuple[str | None, bool | None]:
    try:
        r = http_queue.add('https://character-tavern.com/author/' + username)
        views, downloads, bio, profile_img_url = _extract_user_info_from_sveltekit(r.text)
        if views is None:
            _logger.warning(f'No data for user "{username}"')
            return None, None
        if profile_img_url:
            profile_img_png_bytes = download_file(profile_img_url, ignore_404=True, timeout=GLOBALS.request_timeout_longer)
        else:
            _logger.warning(f'No profile image for user "{username}"')
            profile_img_png_bytes = _DEFAULT_PROFILE_PIC_PATH.read_bytes()
        is_new = _insert_char_tavern_user(username, views, downloads, bio, profile_img_png_bytes)
        return username, is_new
    except Exception as e:
        _logger.error(f'User failed: "{username}" -- {e}\n\n{traceback.format_exc()}')
        return None, None


def process_char_tavern_users(users: Union[List[str], Set[str]], worker_count: int):
    new_users = 0
    with concurrent.futures.ThreadPoolExecutor(max_workers=worker_count) as executor:
        futures = [executor.submit(_process_char_tavern_user, username) for username in sorted(users)]
        for future in concurrent.futures.as_completed(futures):
            username, is_new = future.result()
            end = ''
            if is_new:
                new_users += 1
                end = ' (new)'
            _logger.info(f'{username}{end}')

    return new_users


def _insert_char_tavern_user(username: str, views: int, downloads: int, bio: Union[str, None], png_bytes: bytes):
    try:
        img = strip_exif(Image.open(io.BytesIO(png_bytes))).convert('RGB')
    except ValueError as e:
        _logger.error(f'Failed to convert image "{username}": {e}')
        return

    img.thumbnail(GLOBALS.chub_user_image_max_dimensions, Image.LANCZOS)
    byte_stream = io.BytesIO()
    img.save(byte_stream, format='PNG', optimize=True)

    # Always update the file in case it was missing
    png_hash = write_bytes_to_hashed_file(byte_stream)

    # ===============================================================================
    # SQL

    with CursorFromConnectionFromPool() as cursor:
        # Get the original "added" timestamp
        cursor.execute(
            'SELECT added FROM char_tavern_user WHERE username = %s;',
            (username,)
        )

        is_new = False
        added_result = cursor.fetchone()
        if added_result:
            added_ts = added_result[0]
            cursor.execute(
                """
                UPDATE char_tavern_user
                SET views      = %s,
                    downloads  = %s,
                    bio        = %s,
                    image_hash = %s,
                    updated    = NOW() AT TIME ZONE 'UTC',
                    added      = %s
                WHERE username = %s;
                """,
                (views, downloads, bio, png_hash, added_ts, username)
            )
        else:
            is_new = True
            cursor.execute(
                """
                INSERT INTO char_tavern_user(username, views, downloads, bio, image_hash)
                VALUES (%s, %s, %s, %s, %s);
                """,
                (username, views, downloads, bio, png_hash)
            )
        return is_new
