import tempfile
import traceback
from concurrent.futures import as_completed, ThreadPoolExecutor
from datetime import datetime
from pathlib import Path
from typing import List, Tuple

from scraper import http_queue
from scraper.card_types.ccv2.character import TavernCardv2
from scraper.card_types.ccv3.character import CharacterCardV3
from scraper.character_tavern.download import char_tavern_download
from scraper.character_tavern.insert import insert_char_tavern_character
from scraper.character_tavern.reviews import parse_char_tavern_reviews
from scraper.character_tavern.types import CharTavernNode
from scraper.image import parse_card_metadata
from scraper.log import root_logger

_logger = root_logger.get_child('CHAR-TAVERN.PROCESS')


def _process_char_tavern_node(node: CharTavernNode) -> Tuple[str | None, bool | None, bool | None]:
    try:
        img_bytes = char_tavern_download(node.path)
        if not img_bytes:
            _logger.debug(f'No image for {node.path}')
            return None, None, None

        with tempfile.NamedTemporaryFile() as temp:
            temp.write(img_bytes)
            card_raw, card_parsed, err = parse_card_metadata(Path(temp.name), 'character-tavern')
        if err:
            _logger.error(f'{node.path} -- {err}')
            return None, None, None

        assert card_raw
        assert card_parsed

        # Some cards are missing these dates.
        create_data = datetime.fromtimestamp(node.createdAt)
        card_parsed.data.extensions.char_archive.created = create_data
        if isinstance(card_parsed, CharacterCardV3):
            card_parsed.data.creation_date = node.createdAt
            card_parsed.data.modification_date = node.lastUpdateAt
        elif isinstance(card_parsed, TavernCardv2):
            card_parsed.create_date = create_data

        node_request = http_queue.add(f'https://character-tavern.com/character/{node.path}')
        reviews = {}
        if not node_request or node_request.status_code != 200:
            _logger.error(f'Failed to get reviews for {node.path}: {node_request.status_code or None}')
        else:
            try:
                reviews = parse_char_tavern_reviews(node_request.text)
            except Exception as e:
                _logger.critical(f'Failed to parse reviews for {node.path}: {e}\n{traceback.format_exc()}')

        is_updated, is_new = insert_char_tavern_character(card_parsed, node, img_bytes, card_raw, reviews)
        return node.path, is_updated, is_new
    except Exception as e:
        _logger.error(f'Character failed: "{node.path}" -- {e}\n\n{traceback.format_exc()}')
        return None, None, None


def process_char_tavern_nodes(nodes: List[CharTavernNode], worker_count: int):
    _logger.info('Downloading characters..')
    with ThreadPoolExecutor(max_workers=worker_count) as executor:
        futures = {executor.submit(_process_char_tavern_node, node) for node in nodes}
        for future in as_completed(futures):
            node_path, is_updated, is_new = future.result()
            if node_path:
                mod_str = ''
                if is_updated:
                    mod_str = ' (updated)'
                elif is_new:
                    mod_str = ' (new)'
                _logger.info(f'{node_path}{mod_str}')
            else:
                # If it is None then there has been an error and it has already been handled.
                pass
