import hashlib
import io
import json
import zlib

from PIL import Image
from psycopg2.extras import RealDictCursor

from scraper.card_types.ccv2.character import TavernCardv2
from scraper.card_types.ccv2.funcs import normalize_v2card_for_comparison
from scraper.card_types.funcs import count_char_tokens
from scraper.character_tavern.types import CharTavernNode
from scraper.database.connection import CursorFromConnectionFromPool
from scraper.database.exif import strip_exif
from scraper.database.hash import hash_bytestream
from scraper.globals import GLOBALS
from scraper.insert import write_bytes_to_hashed_file
from scraper.sort import sort_dict


def insert_char_tavern_character(card: TavernCardv2, node: CharTavernNode, png_bytes: bytes, card_raw: str, reviews: dict, extra_metadata: dict = None):
    img = strip_exif(Image.open(io.BytesIO(png_bytes))).convert('RGB')

    # Generate the data hash
    json_bytes = json.dumps(normalize_v2card_for_comparison(sort_dict(json.loads(card.model_dump_json())))).encode('utf-8')
    img_bytes = io.BytesIO()
    img.save(img_bytes, format='PNG')
    m = hashlib.md5()
    m.update(json_bytes)
    m.update(img_bytes.getvalue())
    card_data_hash = m.hexdigest()

    # Finish the image processing
    img.thumbnail(GLOBALS.card_image_max_dimensions, Image.LANCZOS)
    byte_stream = io.BytesIO()
    img.save(byte_stream, format='PNG', optimize=True)

    original_hash = hash_bytestream(io.BytesIO(png_bytes))
    card.data.extensions.char_archive.original_card_hash = original_hash
    card.data.extensions.char_archive.archive_data_hash = card_data_hash

    # Always update the file in case it was missing
    png_hash = write_bytes_to_hashed_file(byte_stream)

    tagline = card.data.scenario

    metadata = {
        'totalTokens': count_char_tokens(card),
    }
    if extra_metadata:
        metadata.update(extra_metadata)

    # ===============================================================================
    # SQL

    with CursorFromConnectionFromPool(cursor_factory=RealDictCursor) as cursor:
        cursor.execute(
            'SELECT * FROM char_tavern_character WHERE author = %s AND path = %s LIMIT 1',
            (node.author, node.path)
        )

        if cursor.fetchone():
            # Already exists
            cursor.execute(
                """
                UPDATE char_tavern_character
                SET name     = %s,
                    data     = %s,
                    reviews  = %s,
                    metadata = %s,
                    updated  = NOW() AT TIME ZONE 'UTC'
                WHERE author = %s
                  AND path = %s;
                """,
                (node.name, node.model_dump_json(), json.dumps(reviews), json.dumps(metadata), node.author, node.path)
            )
        else:
            # Does not exist
            cursor.execute(
                """
                INSERT INTO char_tavern_character(author, name, path, data, reviews, metadata)
                VALUES (%s, %s, %s, %s, %s, %s);
                """,
                (node.author, node.name, node.path, node.model_dump_json(), json.dumps(reviews), json.dumps(metadata))
            )

        cursor.execute(
            'SELECT * FROM char_tavern_character_def WHERE author = %s AND name = %s AND path = %s ORDER BY added DESC LIMIT 1',
            (node.author, node.name, node.path)
        )
        should_insert = False
        existing_cards = 0
        for item in cursor.fetchall():
            existing_cards += 1
            old_card = normalize_v2card_for_comparison(sort_dict(item['definition']))
            new_card = normalize_v2card_for_comparison(sort_dict(json.loads(card.model_dump_json())))
            if old_card != new_card or item['image_hash'] != png_hash:
                should_insert = True
                break

        if should_insert or existing_cards == 0:
            cursor.execute(
                """
                INSERT INTO char_tavern_character_def (author, name, path, definition, raw, image_hash, original_card_hash, metadata)
                VALUES (%s, %s, %s, %s, %s, %s, %s, %s);
                """,
                (node.author, node.name, node.path, card.model_dump_json(), zlib.compress(card_raw.encode()), png_hash, original_hash, json.dumps(metadata))
            )
        return should_insert, existing_cards == 0
