import asyncio
import sys

from playwright.async_api import async_playwright, ProxySettings


async def get_authorization_header():
    async with async_playwright() as p:
        browser = await p.chromium.launch(
            headless=True,
            args=['--disable-blink-features=AutomationControlled'],
            proxy=ProxySettings(server="http://172.0.3.109:9000")
        )
        context = await browser.new_context(
            user_agent='Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36'
        )
        page = await context.new_page()

        authorization_header = None
        request_found = asyncio.Event()

        async def handle_request(request):
            nonlocal authorization_header
            if request.url == 'https://search.character-tavern.com/indexes/characters/search' and request.method == 'POST':
                headers = request.headers
                auth = headers.get('authorization', '')
                if auth:
                    authorization_header = auth
                    request_found.set()

        page.on('request', handle_request)

        response = await page.goto('https://character-tavern.com/character/catalog?search=girl')

        if response.status != 200:
            print(f"ERROR: Page returned status code {response.status} instead of 200", file=sys.stderr)
            await browser.close()
            sys.exit(1)

        try:
            await asyncio.wait_for(request_found.wait(), timeout=30.0)
        except asyncio.TimeoutError:
            print("ERROR: Authorization header not found within timeout period", file=sys.stderr)
            await browser.close()
            sys.exit(1)

        await browser.close()
        return authorization_header


if __name__ == "__main__":
    auth_header = asyncio.run(get_authorization_header())

    if auth_header:
        print(auth_header)
        sys.exit(0)
    else:
        print("ERROR: No authorization header captured", file=sys.stderr)
        sys.exit(1)
