import hashlib
import io
import json
import zlib
from pathlib import Path
from typing import Union

import psycopg2
from PIL import Image

from scraper.card_types.ccv2.character import TavernCardv2
from scraper.card_types.ccv2.funcs import count_v2char_tokens, normalize_v2card_for_comparison
from scraper.database.connection import CursorFromConnectionFromPool
from scraper.database.exif import strip_exif
from scraper.database.hash import hash_bytestream
from scraper.globals import GLOBALS
from scraper.insert import write_bytes_to_hashed_file
from scraper.sort import sort_dict


def insert_generic_character(card: TavernCardv2, png_bytes: bytes, card_raw: str, hashed_data_path: Path, source: str, source_url: Union[str, None], extra_metadata: dict = None):
    img = strip_exif(Image.open(io.BytesIO(png_bytes))).convert('RGB')

    # Generate the data hash
    json_bytes = json.dumps(normalize_v2card_for_comparison(sort_dict(json.loads(card.model_dump_json())))).encode('utf-8')
    img_bytes = io.BytesIO()
    img.save(img_bytes, format='PNG')
    m = hashlib.md5()
    m.update(json_bytes)
    m.update(img_bytes.getvalue())
    card_data_hash = m.hexdigest()

    # Finish the image processing
    img.thumbnail(GLOBALS.card_image_max_dimensions, Image.LANCZOS)
    byte_stream = io.BytesIO()
    img.save(byte_stream, format='PNG', optimize=True)

    original_hash = hash_bytestream(io.BytesIO(png_bytes))
    card.data.extensions.char_archive.original_card_hash = original_hash
    card.data.extensions.char_archive.archive_data_hash = card_data_hash
    # card.data.extensions.char_archive.source_url = source_url

    # Always update the file in case it was missing
    png_hash = write_bytes_to_hashed_file(byte_stream)

    tagline = card.data.scenario

    metadata = {
        'totalTokens': count_v2char_tokens(card),
    }
    # if card.create_date != datetime.fromtimestamp(0):
    #     metadata['created'] = card.create_date.astimezone(timezone.utc).isoformat()
    if extra_metadata:
        metadata.update(extra_metadata)

    # ===============================================================================
    # SQL

    with CursorFromConnectionFromPool() as cursor:
        cursor.execute('SELECT * FROM generic_character_def WHERE source_url = %s LIMIT 1', (source_url,))
        if cursor.fetchone():
            # Duplicate entry, do nothing
            return False

        try:
            cursor.execute(
                """
                INSERT INTO generic_character_def (card_data_hash, name, definition, raw, image_hash, tagline, metadata, source, source_url)
                VALUES (%s, %s, %s, %s, %s, %s, %s, %s, %s);
                """,
                (card_data_hash, card.data.name, card.json(), zlib.compress(card_raw.encode()), png_hash, tagline, json.dumps(metadata), source, source_url)
            )
            return True
        except psycopg2.errors.UniqueViolation as e:
            if 'duplicate key' in str(e).lower():
                # Duplicate entry, do nothing
                return False
            else:
                # An unexpected error occurred
                raise
