import os
import tempfile
from datetime import datetime
from enum import Enum
from pathlib import Path
from types import NoneType
from typing import Tuple, Union
from urllib.parse import urlparse

from dateparser import parse

from proxy_stats.helpers.scrape import is_valid_url
from scraper.card_types.ccv2.character import TavernCardv2
from scraper.card_types.ccv3.character import CharacterCardV3
from scraper.catbox.insert import insert_generic_character
from scraper.catbox.scrape import VALID_EXTENSIONS
from scraper.chub.types.chub_character import ChubCharacter
from scraper.download import download_image
from scraper.image import parse_card_metadata
from scraper.log import root_logger

_logger = root_logger.get_child('CATBOX.DOWNLOAD')


class CatboxDownloadType(Enum):
    CHUB = 0
    GENERIC = 1
    BLANK = 3


def generic_download(url: str, hashed_data_path: Path, allow_non_catbox: bool = False, specific_source: str = None) -> Tuple[str | None, TavernCardv2 | None, bytes | None, str, str | None]:
    assert isinstance(hashed_data_path, Path)
    assert isinstance(allow_non_catbox, bool)
    assert isinstance(specific_source, (str, NoneType))
    assert isinstance(url, str)

    parsed_url = urlparse(url)
    if not is_valid_url(url):
        return None, None, None, url, 'not a valid domain'
    if not allow_non_catbox and parsed_url.netloc not in ['files.catbox.moe', 'litterbox.catbox.moe', 'litter.catbox.moe']:
        return None, None, None, url, 'not an allowed domain'

    _, ext = os.path.splitext(url)
    if ext in VALID_EXTENSIONS:
        parsed_url = urlparse(url)
        img_bytes, err, response = download_image(url)
        if err and err.startswith('404'):
            if parsed_url.netloc in ['litterbox.catbox.moe', 'litter.catbox.moe']:
                _logger.debug(f'Litterbox 404: {url}')
                return None, None, None, url, None
            else:
                return None, None, None, url, err
        if not img_bytes:
            return None, None, None, url, err

        with tempfile.NamedTemporaryFile() as temp:
            temp.write(img_bytes)
            card_raw, card_parsed, err = parse_card_metadata(Path(temp.name), 'catbox' if not specific_source else specific_source)
        if err:
            return None, None, None, url, err
        assert card_raw
        assert card_parsed

        if isinstance(card_parsed, CharacterCardV3):
            return None, None, None, url, f'Ignoring CCv3 Card: {url}'

        if '.catbox.moe' in parsed_url.netloc and card_parsed.create_date == datetime.fromtimestamp(0):
            _logger.debug(f'Card create date: {card_parsed.create_date}. Last-Modified header: {parse(response.headers.get("last-modified"))} (using header) -> "{url}"')
            card_parsed.create_date = parse(response.headers.get('last-modified'))

        if not card_raw or not card_parsed:
            return None, None, None, url, err
        return card_raw, card_parsed, img_bytes, url, None
    else:
        return None, None, None, url, 'not a png'


def catbox_download_and_import_card(hashed_data_path: Path, url: str, specific_source: str = None, allow_non_catbox: bool = False, extra_metadata: Union[dict, None] = None, perform_insert: bool = True) -> Tuple[CatboxDownloadType, ChubCharacter | None, bool, bool, str, str | None]:
    assert url is not None

    if not extra_metadata:
        extra_metadata = {}
    # extra_metadata['source_url'] = url

    card_raw, card_parsed, img_bytes, _, err = generic_download(url, hashed_data_path, allow_non_catbox=allow_non_catbox, specific_source=specific_source)
    if card_raw is None or card_parsed is None or img_bytes is None or err is not None:
        return CatboxDownloadType.BLANK, None, False, False, url, err

    # if card_parsed.data.extensions.model_extra.get('chub'):
    #     if perform_insert:
    #         is_new, is_updated, char = catbox_insert_chub_card(card_parsed, card_raw, img_bytes, hashed_data_path, extra_metadata=extra_metadata)
    #         return CatboxDownloadType.CHUB, char, is_new, is_updated, url, None
    #     return CatboxDownloadType.CHUB, None, False, False, url, None
    # else:
    if perform_insert:
        is_new = insert_generic_character(card_parsed, img_bytes, card_raw, hashed_data_path, 'catbox' if not specific_source else specific_source, url, extra_metadata=extra_metadata)
        return CatboxDownloadType.GENERIC, None, is_new, False, url, None
    return CatboxDownloadType.BLANK, None, False, False, url, None
