from datetime import datetime, timezone
from typing import Optional, Union

from dateparser import parse
from pydantic import BaseModel, field_validator, Field


def parse_date_field(v) -> datetime | None:
    """
    Parse any date field in an item.
    Does not automatically parse the strange Tavern format (use `st_timestamp_to_datetime()`).
    :param v:
    :return:
    """
    dt = None
    if isinstance(v, (int, float)):
        if v > 9999999999:  # Assuming timestamp is in milliseconds
            v = v / 1000
        dt = datetime.fromtimestamp(v)
    elif isinstance(v, datetime):
        dt = v
    elif isinstance(v, str):
        dt = parse(v)
    if dt:
        return dt.astimezone(timezone.utc)
    else:
        return None


class CharArchiveExtension(BaseModel):
    """
    For V2 and V3 cards.
    """

    created: Optional[datetime] = None
    added: Optional[datetime] = Field(default_factory=lambda: datetime.now(timezone.utc))
    archive_data_hash: Optional[str] = None
    original_card_hash: Optional[str] = None
    source: Optional[str] = None
    source_url: Optional[str] = None
    source_id: Optional[Union[str, int]] = None

    class Config:
        json_encoders = {
            datetime: lambda v: v.astimezone(timezone.utc).isoformat()
        }
        validate_assignment = True

    @field_validator('added', mode='before')
    @classmethod
    def parse_added(cls, v):
        return parse_date_field(v)

    @field_validator('created', mode='before')
    @classmethod
    def parse_created(cls, v):
        if v is not None:
            return parse_date_field(v)
        else:
            return None


class ItemExtensions(BaseModel):
    char_archive: Optional[CharArchiveExtension] = Field(default_factory=CharArchiveExtension)

    class Config:
        extra = 'allow'
