from types import NoneType
from typing import List, Optional, Union

import tiktoken
from pydantic import BaseModel, Field

from scraper.card_types.item_extensions import ItemExtensions


class CharacterCardV3LorebookEntry(BaseModel):
    keys: List[str]
    content: str
    enabled: bool
    insertion_order: Optional[int] = None  # Supposed to be an int but some cards have this set as `null`
    use_regex: Optional[bool] = False
    extensions: dict = Field(default_factory=dict)
    case_sensitive: Optional[bool] = None
    constant: Optional[bool] = None
    name: Optional[str] = None
    priority: Optional[int] = None
    id: Optional[Union[int, str]] = None
    comment: Optional[str] = None
    selective: Optional[bool] = None
    secondary_keys: Optional[List[str]] = None
    position: Union[str, int, NoneType] = None

    class Config:
        validate_assignment = True


class CharacterCardV3Lorebook(BaseModel):
    entries: List[CharacterCardV3LorebookEntry]
    name: Optional[str] = None
    description: Optional[str] = None
    extensions: Optional[ItemExtensions] = Field(default_factory=ItemExtensions)
    scan_depth: Optional[int] = None
    token_budget: Optional[int] = None
    recursive_scanning: Optional[bool] = None
    creation_date: Optional[int] = 0
    modification_date: Optional[int] = 0

    class Config:
        validate_assignment = True


class CharacterCardV3LorebookRoot(BaseModel):
    spec: str = 'lorebook_v3'
    data: CharacterCardV3Lorebook


def count_v3lorebook_tokens(card: CharacterCardV3Lorebook):
    text_stuff = ''.join([x.content for x in card.entries])
    return len(tiktoken.get_encoding("cl100k_base").encode(text_stuff, disallowed_special=()))
