import copy

from scraper.card_types.ccv1.character import TavernCardv1
from scraper.card_types.ccv2.character import TavernCardv2, TavernCardv2Data
from scraper.card_types.ccv2.lorebook import normalize_lorebook_entries
from scraper.card_types.funcs import test_card_name
from scraper.card_types.item_extensions import CharArchiveExtension
from scraper.card_types.st_timestamp import parse_st_create_date_timestamp


def normalize_cardv2(raw_card: dict, author: str, source: str):
    assert source is not None
    full_card = copy.deepcopy(raw_card)
    del raw_card
    definition = full_card['data']
    name = definition['name']
    if not test_card_name(definition['name']) or not test_card_name(name):
        raise ValueError('Character name is empty')

    definition['creator'] = author if author is not None else definition['creator']
    if definition.get('char_name'):
        del definition['char_name']
    if definition.get('personality'):
        if not definition.get('char_persona'):
            definition['char_persona'] = definition['personality']
        del definition['personality']
    if definition.get('alternate_greetings') and not isinstance(definition['alternate_greetings'], list):
        del definition['alternate_greetings']

    # Character Tavern cards might be missing the `scenario` key. If so, copy the description to it.
    if not definition.get('scenario') and definition.get('description'):
        definition['scenario'] = definition['description']

    if 'events' in list(definition.keys()):
        raise Exception

    # Create the char archive extension while preserving existing data char archive extension data.
    if not full_card['data'].get('extensions'):
        full_card['data']['extensions'] = {}
    if not full_card['data']['extensions'].get('char_archive'):
        full_card['data']['extensions']['char_archive'] = {}
    full_card['data']['extensions']['char_archive']['source'] = source
    if full_card.get('create_date'):
        full_card['data']['extensions']['char_archive']['created'] = parse_st_create_date_timestamp(full_card['create_date'])
    full_card['data']['extensions']['char_archive'] = CharArchiveExtension(**full_card['data']['extensions']['char_archive']).model_dump()

    if 'chat' in list(definition.keys()):
        del definition['chat']

    if definition.get('character_book'):
        definition['character_book']['entries'] = normalize_lorebook_entries(definition['character_book']['entries'])

    card = TavernCardv2(data=definition, create_date=full_card.get('create_date'), creatorcomment=full_card.get('creatorcomment', ''))
    if card.data.character_book:
        del card.data.character_book.extensions.char_archive
        if len(card.data.character_book.extensions.model_dump().keys()) == 0:
            del card.data.character_book.extensions
        del card.data.character_book.create_date

    return card


def convert_spec1_to_spec2(card: TavernCardv1, author: str, source: str):
    data = {
        'name': card.name,
        'scenario': card.scenario,
        'first_mes': card.first_mes,
        'mes_example': card.mes_example,
        'description': card.description,
        'personality': card.personality,
        'creator': author
    }
    v2_card = TavernCardv2(data=TavernCardv2Data(**data))
    return normalize_cardv2(v2_card.model_dump(), author, source)


def infer_specv2_card(raw_card: dict, source: str, specific_author: str = None):
    if not specific_author or specific_author == '':
        specific_author = raw_card.get('data', {}).get('author')
    full_card = raw_card.copy()
    if 'data' in list(full_card.keys()):
        return normalize_cardv2(raw_card, specific_author, source)
    else:
        return convert_spec1_to_spec2(TavernCardv1(**full_card), specific_author, source)
