from datetime import datetime, timezone
from typing import Optional, List

from pydantic import BaseModel, Field, field_validator

from scraper.card_types.ccv2.lorebook import LorebookV2
from scraper.card_types.item_extensions import ItemExtensions
from scraper.card_types.st_timestamp import parse_st_create_date_timestamp


class TavernCardv2Data(BaseModel):
    name: str
    scenario: str
    first_mes: str
    creator: Optional[str] = ''
    mes_example: Optional[str] = ''
    description: Optional[str] = ''
    personality: Optional[str] = ''
    creator_notes: Optional[str] = ''
    system_prompt: Optional[str] = ''
    post_history_instructions: Optional[str] = ''
    alternate_greetings: List[str] = Field(default_factory=list)
    character_book: Optional[LorebookV2] = None
    tags: List[str] = Field(default_factory=list)
    character_version: Optional[str] = ''
    extensions: Optional[ItemExtensions] = Field(default_factory=ItemExtensions)
    avatar: Optional[str] = None
    example_dialogue: Optional[str] = ''
    world_scenario: Optional[str] = ''

    class Config:
        validate_assignment = True


class TavernCardv2(BaseModel):
    spec: str = 'chara_card_v2'
    spec_version: str = '2.0'
    data: TavernCardv2Data
    create_date: Optional[datetime] = Field(default_factory=lambda: datetime.fromtimestamp(0))
    creatorcomment: Optional[str] = ''

    class Config:
        json_encoders = {
            datetime: lambda v: v.astimezone(timezone.utc).isoformat()
        }
        validate_assignment = True

    @field_validator('create_date', mode='before')
    @classmethod
    def parse_date(cls, v):
        return parse_st_create_date_timestamp(v)
