import io
import json
import zlib
from datetime import timezone, datetime
from json import JSONEncoder
from pathlib import Path

from PIL import Image
from psycopg2.extras import RealDictCursor

from scraper.booru.parse import BooruNode
from scraper.database.connection import CursorFromConnectionFromPool
from scraper.database.exif import strip_exif
from scraper.database.hash import hash_bytestream
from scraper.globals import GLOBALS
from scraper.insert import write_bytes_to_hashed_file
from scraper.card_types.ccv2.funcs import count_v2char_tokens
from scraper.card_types.ccv2.character import TavernCardv2


class DateTimeEncoder(JSONEncoder):
    def default(self, o):
        if isinstance(o, datetime):
            return o.astimezone(timezone.utc).isoformat()

        return super(DateTimeEncoder, self).default(o)


def insert_booru_character(node: BooruNode, card: TavernCardv2, png_bytes: bytes, card_raw: str, hashed_data_path: Path):
    img = strip_exif(Image.open(io.BytesIO(png_bytes))).convert('RGB')
    img.thumbnail(GLOBALS.card_image_max_dimensions, Image.LANCZOS)
    byte_stream = io.BytesIO()
    img.save(byte_stream, format='PNG', optimize=True)

    original_hash = hash_bytestream(io.BytesIO(png_bytes))
    card.data.extensions.char_archive.original_card_hash = original_hash

    png_hash = write_bytes_to_hashed_file(byte_stream)

    card.data.extensions.char_archive.source_id = node.bpi
    card.data.extensions.char_archive.original_card_hash = hash_bytestream(io.BytesIO(png_bytes))
    card.data.extensions.char_archive.created = node.date
    card.create_date = node.date

    metadata = {
        'totalTokens': count_v2char_tokens(card),
        'safe': node.rating
    }
    comments_json = json.dumps(node.dict()['comments'], cls=DateTimeEncoder)

    # ===============================================================================
    # SQL

    with CursorFromConnectionFromPool(cursor_factory=RealDictCursor) as cursor:
        cursor.execute(
            """
            INSERT INTO booru_character_def (name, author, id, definition, raw, comments, tags, image_hash, tagline, metadata, created)
            VALUES (%s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s);
            """,
            (card.data.name, node.author, node.bpi, card.json(), zlib.compress(card_raw.encode()), comments_json, node.tags, png_hash, card.data.scenario, json.dumps(metadata), node.date)
        )
