#!/usr/bin/env python3
import argparse
import copy
import json

from psycopg2.extras import RealDictCursor
from tqdm import tqdm

from scraper.card_types.ccv2.character import TavernCardv2
from scraper.card_types.ccv2.funcs import count_v2char_tokens
from scraper.card_types.ccv2.infer import infer_specv2_card
from scraper.card_types.ccv3.character import CharacterCardV3
from scraper.card_types.ccv3.funcs import count_v3char_tokens
from scraper.card_types.ccv3.infer import infer_specv3_card
from scraper.database.connection import Database, CursorFromConnectionFromPool


def process_table(rows, table_name, primary_key: tuple):
    for row in tqdm(rows, desc=table_name):
        card_def = row['definition']
        if card_def['spec'] == '2.0':
            card = infer_specv2_card(card_def, '', '')
        else:
            try:
                card = infer_specv3_card(card_def, '')
            except:
                print(card_def)
                raise
        if isinstance(card, CharacterCardV3):
            token_count = count_v3char_tokens(card)
        elif isinstance(card, TavernCardv2):
            token_count = count_v2char_tokens(card)
        else:
            raise Exception

        metadata = copy.deepcopy(row['metadata'])
        metadata['totalTokens'] = token_count

        with CursorFromConnectionFromPool(cursor_factory=RealDictCursor) as cursor:
            cursor.execute(f"UPDATE {table_name} SET metadata = %s WHERE {' AND '.join([f'{k} = %s' for k in primary_key])}", (json.dumps(metadata), *[row[k] for k in primary_key]))

        tqdm.write(f'{row[primary_key[0]]} -- {card.data.name} -- {token_count}')


def main(args):
    Database.initialise(minconn=1, maxconn=100, host='172.0.3.101', database='char_archive', user='char_archive', password='hei3ucheet5oochohjongeisahV3mei0')

    with CursorFromConnectionFromPool(cursor_factory=RealDictCursor) as cursor:
        print('Fetching chub')
        cursor.execute('SELECT * FROM chub_character_def' + (' LIMIT 20' if args.test else ''))
        process_table(cursor.fetchall(), 'chub_character_def', ('id', 'increment'))

        print('Fetching booru')
        cursor.execute('SELECT * FROM booru_character_def' + (' LIMIT 20' if args.test else ''))
        process_table(cursor.fetchall(), 'booru_character_def', ('id',))

        print('Fetching generic')
        cursor.execute('SELECT * FROM generic_character_def' + (' LIMIT 20' if args.test else ''))
        process_table(cursor.fetchall(), 'generic_character_def', ('card_data_hash', 'increment'))

        print('Fetching nyaime')
        cursor.execute('SELECT * FROM nyaime_character_def' + (' LIMIT 20' if args.test else ''))
        process_table(cursor.fetchall(), 'nyaime_character_def', ('id', 'increment'))

        print('Fetching risuai')
        cursor.execute('SELECT * FROM risuai_character_def' + (' LIMIT 20' if args.test else ''))
        process_table(cursor.fetchall(), 'risuai_character_def', ('id', 'increment'))


if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument('-t', '--test', action='store_true')
    args = parser.parse_args()
    main(args)
