import logging
import os
import sys
import time
import traceback
from datetime import datetime, timedelta
from pathlib import Path
from urllib.parse import urlparse

import coloredlogs
import elastic_transport
import requests
from flask import Flask, jsonify, request
from flask_caching import Cache
from flask_sqlalchemy import SQLAlchemy
from sqlalchemy import Column, String
from sqlalchemy.exc import IntegrityError

script_dir = os.path.dirname(os.path.realpath(__file__))
sys.path.insert(0, str(Path(script_dir).parent))

from helpers.cache import cache_control
from helpers.helpers import check_int, validate_unix_timestamp
from helpers.parse import parse_proxy_stats, bump_token_usage
from helpers.elastic import ELASTIC_CLIENT
from helpers.regex import remove_http
from helpers.scrape import is_valid_url, validate_proxy
from helpers.settings import BANNED_PROXY_HOSTS

app = Flask(__name__)
cache = Cache(app, config={'CACHE_TYPE': 'redis', 'CACHE_REDIS_URL': 'redis://localhost:6379/0', 'CACHE_KEY_PREFIX': 'proxy_stats__'})
cache.clear()

DATABASE_IP = os.environ.get('DATABASE_IP', '127.0.0.1')
app.config['SQLALCHEMY_DATABASE_URI'] = f'postgresql://char_archive:hei3ucheet5oochohjongeisahV3mei0@{DATABASE_IP}/proxy_stats'
alchemy_db = SQLAlchemy(app)

logging.basicConfig()
logger = logging.getLogger('SERVER')
logger.setLevel(logging.INFO)
coloredlogs.install(logger=logger, level=logging.INFO)


class Proxies(alchemy_db.Model):
    __tablename__ = 'proxies'
    url = Column(String, primary_key=True)


class ApiTypes(alchemy_db.Model):
    __tablename__ = 'api_types'
    type = Column(String, primary_key=True)


while True:
    try:
        ELASTIC_CLIENT.connect('https://172.0.3.105:9200', 'proxy_stats', 'RU9FNWRZOEJod0d0ZGl3YjlRMGM6NS1OV0dTbl9ROGVDZkNqSnJId3c0Zw==')
        break
    except elastic_transport.ConnectionError as e:
        logger.error(f'Failed to connect to Elasticsearch: {e}')
        time.sleep(30)


@app.route('/api/list', methods=['GET'])
@cache_control(3600)
@cache.cached(timeout=3600, query_string=True)
def get_tables():
    tables = sorted([row.url for row in Proxies.query.all()])
    valid = []
    for proxy in tables:
        latest_query = ELASTIC_CLIENT.query(field_value_pairs={'url': proxy}, max_results=1)
        if len(latest_query) and latest_query[0]['_source'].get('timestamp'):
            t = datetime.fromtimestamp(latest_query[0]['_source']['timestamp'])
            if t > datetime.now() - timedelta(days=14):
                valid.append(proxy)
    return jsonify(valid)


@app.route('/api/types', methods=['GET'])
@cache_control(3600)
@cache.cached(timeout=3600, query_string=True)
def get_api_types():
    tables = sorted([row.type for row in ApiTypes.query.all()])
    return jsonify(tables)


@app.route('/api/add', methods=['POST'])
def add_proxy():
    url = request.get_json().get('url')
    if not url:
        return 'No URL provided', 400
    if not is_valid_url(url):
        return 'Invalid URL provided', 400
    for bad in BANNED_PROXY_HOSTS:
        parsed_url = urlparse(url)
        if bad in parsed_url.hostname:
            return 'This URL is not allowed', 400
    bare_url = remove_http(url)
    try:
        existing_proxy = Proxies.query.filter_by(url=bare_url).first()
        if existing_proxy:
            return 'Proxy already exists', 400

        try:
            response = requests.get(url, timeout=10, headers={'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36'}, proxies={'http': 'http://172.0.3.109:9000', 'https': 'http://172.0.3.109:9000'})
            if response.status_code != 200:
                return 'Failed to reach remote', 400
            proxy_json, proxy_url, err_msg = validate_proxy(url, response)
            if not proxy_json:
                return 'Not an OAI reverse proxy', 400
        except requests.exceptions.ConnectionError:
            return 'Failed to reach proxy', 400

        try:
            new_proxy = Proxies(url=bare_url)
            alchemy_db.session.add(new_proxy)
            alchemy_db.session.commit()
        except IntegrityError:
            alchemy_db.session.rollback()
            return 'Proxy already exists', 400
    except:
        logger.error(traceback.print_exc())
        return 'Internal server error', 500
    return 'New proxy added'


@app.route('/api/get', methods=['GET'])
@cache_control(60)
@cache.cached(timeout=60, query_string=True)
def get_proxy_data():
    proxy = remove_http(request.args.get('proxy'))
    start_arg = request.args.get('start')
    to_arg = request.args.get('to')
    raw_json_arg = request.args.get('rawJson')
    prettify_arg = request.args.get('prettify')
    historical_arg = request.args.get('historical')
    zoom_level_arg = request.args.get('zoomLevel')

    if not start_arg and to_arg:
        return jsonify({
            'code': 400,
            'msg': 'must supply start parameter when using to parameter'
        }), 400

    if not proxy:
        return jsonify({
            'code': 400,
            'msg': 'missing proxy parameter'
        }), 400

    if zoom_level_arg and not check_int(zoom_level_arg):
        return jsonify({
            'code': 400,
            'msg': 'zoom level must be an integer'
        }), 400

    @cache.cached(timeout=60, query_string=True)
    def fetch_data():
        nonlocal start_arg, to_arg, raw_json_arg, prettify_arg, historical_arg

        # Replace invalid characters in the proxy name
        table_name = proxy.replace('.', '_').replace('/', '_')

        # Check if the table exists
        tables = Proxies.query.filter(Proxies.url.like(f'{table_name}%')).all()
        if not len(tables):
            return jsonify({'code': 'Proxy does not exist'}), 400
        if len(tables) > 1:
            return jsonify({'code': 'Multiple proxies returned, refine your search', 'matches': [x[0] for x in tables]}), 400
        proxy_url = tables[0].url

        # If start and to parameters are not supplied, get data from the last 12 hours
        if to_arg:
            if not validate_unix_timestamp(to_arg):
                return jsonify({
                    'code': 400,
                    'msg': 'to parameter is invalid'
                }), 400
            to = int(to_arg)
        else:
            to = int(time.time())
        if start_arg:
            if not validate_unix_timestamp(start_arg):
                return jsonify({
                    'code': 400,
                    'msg': 'start parameter is invalid'
                }), 400
            start = int(start_arg)
        else:
            start = to - 6 * 60 * 60  # 6 hours ago

        try:
            latest_query = ELASTIC_CLIENT.query(field_value_pairs={'url': proxy_url}, max_results=1)
        except elastic_transport.ConnectionTimeout:
            return jsonify({
                'code': 500,
                'msg': 'failed to reach Elasticsearch server'
            }), 500

        if not len(latest_query):
            return jsonify({
                'error': 'no data for selected proxy',
            }), 400
        latest_stats = parse_proxy_stats(latest_query[0]['_source'])

        if raw_json_arg != 'true':
            del latest_stats['raw_json']

        if prettify_arg == 'true':
            for k, v in latest_stats['apiTypes'].items():
                try:
                    if isinstance(latest_stats['apiTypes'][k]['usage'], dict):
                        vv = bump_token_usage(latest_stats['apiTypes'][k]['usage'])
                        cost_str = str(vv["cost"])
                        if cost_str.endswith('.0'):
                            cost_str = cost_str + '0'
                        latest_stats['apiTypes'][k]['usage'] = f'{vv["tokens"]} (${cost_str})'
                except:
                    latest_stats['apiTypes'][k]['usage'] = '0 ($0.00)'

            vv = bump_token_usage(latest_stats['stats']['tookens'])
            cost_str = str(vv["cost"])
            if cost_str.endswith('.0'):
                cost_str = cost_str + '0'
            latest_stats['stats']['tookens'] = f'{vv["tokens"]} (${cost_str})'

        response = {
            'latest': latest_stats,
        }

        if historical_arg != 'false':
            try:
                historical_query = ELASTIC_CLIENT.query(field_value_pairs={'url': proxy_url}, start_arg=start, end_arg=to)
            except elastic_transport.ConnectionTimeout:
                return jsonify({
                    'code': 500,
                    'msg': 'failed to reach Elasticsearch server'
                }), 500

            historical_stats = {}
            for item in historical_query:
                parsed_data = parse_proxy_stats(item['_source'])
                ts = parsed_data['timestamp']
                del parsed_data['raw_json']
                del parsed_data['build']
                del parsed_data['config']
                del parsed_data['endpoints']
                del parsed_data['timestamp']
                del parsed_data['url']

                # This is not a number.
                for api_type, data in parsed_data['apiTypes'].items():
                    for k, v in data.items():
                        if k == 'status':
                            del parsed_data['apiTypes'][api_type]['status']
                            break

                historical_stats[ts] = parsed_data

            # Reduce the number of points based on the zoom level
            if zoom_level_arg:
                zoom_level = int(zoom_level_arg)
                keys = sorted(historical_stats.keys())
                reduced_stats = {k: historical_stats[k] for k in keys[::zoom_level]}
                response['historical'] = reduced_stats
            else:
                response['historical'] = historical_stats

        return jsonify(response)

    return fetch_data()


@app.route('/')
@app.route('/<first>')
@app.route('/<first>/<path:rest>')
def fallback(first=None, rest=None):
    return jsonify({
        'code': 404,
        'msg': 'not an endpoint'
    }), 400


@app.errorhandler(500)
def server_error(e):
    app.logger.error(e)
    return jsonify({
        'code': 500,
        'msg': 'internal server error'
    }), 500


if __name__ == '__main__':
    app.run(debug=True, host='0.0.0.0')
