import copy
import json
import re
from typing import Union

from scraper.log import root_logger
from .api import API_TYPES
from .helpers import check_type

logger = root_logger.get_child('PROXY_STATS.PARSE')


def parse_usage(usage_str: str) -> dict[str, float]:
    if isinstance(usage_str, dict):
        return usage_str
    usage_split = [x.strip(' ') for x in usage_str.split(' ')]
    for i, x in enumerate(usage_split.copy()):
        if x == 'tokens':
            del usage_split[i]
    if len(usage_split) == 2:
        int_str, money_str = usage_split
        money_str = money_str.strip('($').strip(')')
        return {'tokens': float(convert_usage_int(int_str)), 'cost': float(money_str)}
    elif len(usage_split) == 1:
        if '$' in usage_split[0]:
            usage = float(usage_split[0].strip('($').strip(')'))
            return {'tokens': float(-1), 'cost': float(usage)}
        else:
            usage = convert_usage_int(usage_split[0])
            return {'tokens': float(usage), 'cost': float(-1)}


def parse_proxy_stats(item: dict):
    # Don't modify the original item.
    item = copy.deepcopy(item)

    try:
        raw_json = json.loads(item.get('json'))
    except:
        raw_json = None

    result = {
        'apiKeys': {k.replace('Keys', ''): v for k, v in item.items() if k.endswith('Keys')},
        'apiTypes': {},
        'stats': {
            'uptime': item.get('uptime'),
            'proompts': item.get('proompts', 0),
            'tookens': parse_usage(item.get('tookens')) if isinstance(item.get('tookens'), str) else item.get('tookens', 0),
            'proomptersNow': item.get('proomptersNow', 0)
        },
        'config': item.get('config', {}),
        'build': item.get('build'),
        'timestamp': item.get('timestamp'),
        'raw_json': raw_json,
        'endpoints': item.get('endpoints', {}),
        'url': 'https://' + item['url'] if 'url' in item.keys() else None
    }

    for api_type in API_TYPES:
        api_type = api_type.replace('*', '.*?')  # Expand any wildcards.
        for k, v in item.items():
            api_type_name = k.replace('-', '_')
            if '*' not in api_type and api_type == k:
                # If no wildcard was specified and there was an exact match.
                result['apiTypes'][api_type_name] = v
            elif '*' in api_type and re.match(api_type, k):
                # If a wildcard was specified and there was a match.
                if isinstance(v, dict):
                    # There may be other non-dict variables in the proxy's JSON response that we will ignore.
                    result['apiTypes'][api_type_name] = v

    for api_type, v in result['apiTypes'].items():
        result['apiTypes'][api_type]['usage'] = parse_usage(v.get('usage', '0 ($0.00)'))

    return result


def convert_usage_int(s):
    if check_type(s, float):
        return float(s)
    match = re.match(r"([0-9.]+)([a-z]+)", s, re.I)
    if match:
        items = match.groups()
    else:
        # Something broke, just fall back to 0.
        logger.warn(f'Something went wrong, setting usage "{s}" to 0.')
        return 0
    if 'k' in items[1]:
        return int(float(items[0]) * 1e3)
    elif 'm' in items[1]:
        return int(float(items[0]) * 1e6)
    elif 'b' in items[1]:
        return int(float(items[0]) * 1e9)
    elif 't' in items[1]:
        return int(float(items[0]) * 1e12)


def bump_token_usage(usage: dict):
    if usage["tokens"] == -1:
        usage["tokens"] = 0
    if usage["cost"] == -1:
        usage["cost"] = 0.0
    return usage


def fix_question(field):
    if field == '???':
        logger.warn('Converted "???" to 0.')
        return 0
    else:
        return field


def fix_all_questions(d: Union[dict, list, tuple]):
    for k, v in d.copy().items():
        if isinstance(v, str):
            d[k] = fix_question(v)
        elif isinstance(v, dict):
            d[k] = fix_all_questions(v)
        elif isinstance(v, list):
            for i, vv in enumerate(v):
                d[k][i] = fix_question(vv)
        elif isinstance(v, tuple):
            d[k] = list(v)
            for i, vv in enumerate(v):
                d[k][i] = fix_question(vv)
    return d


def extract_api_types(proxy_json: dict):
    # Read the specified API card_types.
    types = []
    for api_type in API_TYPES:
        api_type = api_type.replace('*', '.*?')  # Expand any wildcards.
        for k, v in proxy_json.items():
            api_type_name = k.replace('-', '_')
            if '*' not in api_type and api_type == k:
                # If no wildcard was specified and there was an exact match.
                types.append(api_type_name)
            elif '*' in api_type and re.match(api_type, k):
                # If a wildcard was specified and there was a match.
                types.append(api_type_name)
    return types
