import time
import warnings

import elastic_transport
from elasticsearch import Elasticsearch

from scraper.log import root_logger

logger = root_logger.get_child('PROXY_STATS.ELASTIC')


class ElasticClient:
    def __init__(self):
        self.es = Elasticsearch(['http://localhost:9200'])

    def connect(self, elastic_host: str, elastic_index: str, api_key: str):
        self.elastic_host = elastic_host
        self.elastic_index = elastic_index
        self.api_key = api_key

        with warnings.catch_warnings(action='ignore'):
            self.es = Elasticsearch(
                [self.elastic_host],
                api_key=self.api_key,
                verify_certs=False
            )
            logger.debug(f'Connected to Elastic: {self.es.info()}')
        return self.es

    def create_index(self, template: dict = None):
        with warnings.catch_warnings(action='ignore'):
            if not self.es.indices.exists(index=self.elastic_index):
                self.es.indices.create(index=self.elastic_index, body=template)

    def insert_json(self, json_data: dict, item_id: str = None):
        for i in range(3):
            try:
                with warnings.catch_warnings(action='ignore'):
                    return self.es.index(index=self.elastic_index, body=json_data, id=item_id)
            except elastic_transport.ConnectionTimeout:
                logger.warning(f'Elastic timeout. Retry: {i}')
                time.sleep(5)
            except:
                print(json_data)
                raise

    def query(self, query_str: str = None, start_arg: int = None, end_arg: int = None, field_value_pairs: dict = None, sort_timestamp_field: bool = True, max_results: int = 10000, timeout: int = 30):
        if query_str and field_value_pairs:
            raise ValueError('Cannot specify both query_str and field_value_pairs')

        body = {
            "query": {
            }
        }

        if max_results:
            body['size'] = max_results

        if query_str:
            body['query'] = {
                "bool": {
                    "must": [
                        {
                            "query_string": {
                                "query": query_str
                            }
                        },
                    ]
                }
            }
        elif field_value_pairs:
            body["query"]["bool"] = {'must': []}
            for field, value in field_value_pairs.items():
                body["query"]["bool"]["must"].append({
                    "term": {
                        f'{field}.keyword': value
                    }
                })
        else:
            raise ValueError('Must specify either query_str or field_value_pairs')

        if start_arg and end_arg:
            # The timestamp args must be epoch seconds.
            if not body["query"].get("bool", {}).get("must", []):
                body["query"]["bool"] = {'must': []}
            body["query"]["bool"]["must"].append({
                "range": {
                    "timestamp": {
                        "gte": start_arg * 1000,  # The epoch_second mapping requires timestamp values in milliseconds.
                        "lte": end_arg * 1000
                    }
                }
            })

        if sort_timestamp_field:
            body['sort'] = [
                {
                    "timestamp": {
                        "order": "desc"
                    }
                }
            ]

        with warnings.catch_warnings(action='ignore'):
            response = self.es.search(index=self.elastic_index, body=body, request_timeout=timeout)
        return response['hits']['hits']


ELASTIC_CLIENT = ElasticClient()
