#!/usr/bin/env python3
import argparse
import logging
import os
import sys
import traceback
import yaml
from pathlib import Path
from tqdm import tqdm

from scraper import log
from scraper.config.models import ConfigModel
from scraper.database.connection import Database
from scraper.paths import resolve_path
from scraper.suicide import watchdog_suicide

script_dir = os.path.dirname(os.path.realpath(__file__))
sys.path.insert(0, str(Path(script_dir).parent))

from scraper.image import parse_card_metadata
from scraper.catbox.insert import insert_generic_character

"""
This script imports generic cards.
"""

_SCRIPT_DIR = os.path.dirname(os.path.realpath(__file__))


def process_catbox(card_path: Path, hashed_data_path: Path, source: str):
    img_bytes = card_path.read_bytes()
    card_raw, card_parsed, err = parse_card_metadata(card_path, source, only_exceptions=True)
    if err:
        print(err)
        quit(1)
    if not card_parsed:
        return

    return insert_generic_character(card_parsed, img_bytes, card_raw, hashed_data_path, source, source_url=None)


def main(args):
    cfg_data = yaml.load(resolve_path(args.config).read_text(), Loader=yaml.FullLoader)
    cfg = ConfigModel(**cfg_data)

    global _logger
    if args.debug:
        log_level = logging.DEBUG
    else:
        log_level = logging.INFO
    log.root_logger.init(log_level)
    _logger = log.root_logger.logger

    args.target_path = resolve_path(args.target_path)
    if not args.target_path.is_dir():
        _logger.critical(f'Path {args.target_path} is not a directory.')
        watchdog_suicide()

    Database.initialise(minconn=1, maxconn=100, host=cfg.database.host, database=cfg.database.database, user=cfg.database.user, password=cfg.database.password)

    files = sorted(list(args.target_path.glob('*.png')))
    catbox_loading_bar = tqdm(total=len(files), desc='Importing Cards', smoothing=0.05, leave=True)
    new_cards = 0
    for p in files:
        catbox_loading_bar.write(str(p))
        new = process_catbox(p, cfg.hashed_data_path, args.source)
        if new:
            new_cards += 1
        catbox_loading_bar.update(1)
    catbox_loading_bar.close()
    print('Total cards imported:', new_cards)


if __name__ == '__main__':
    parser = argparse.ArgumentParser(description='')
    parser.add_argument('target_path', help='The path to import.')
    parser.add_argument('source', help='The source of these cards.')
    parser.add_argument('--config', default=Path(_SCRIPT_DIR, '../config', 'chub.yml'), help='Path to the config file. Default: config/chub.yml in the config directory next to the script.')
    parser.add_argument('-d', '--debug', action='store_true', help='Enable debug logging.')
    args = parser.parse_args()
    try:
        main(args)
    except:
        traceback.print_exc()
        sys.exit(1)
