import io
import json
import os
import random
import sys
from json import JSONDecodeError
from pathlib import Path

import requests
from PIL import Image
from pydantic import ValidationError
from tqdm import tqdm

script_dir = os.path.dirname(os.path.realpath(__file__))
sys.path.insert(0, str(Path(script_dir).parent))

from scraper.chub.chub import CHUB_TYPE_STR
from scraper.card_types.ccv2.lorebook import normalize_lorebook
from scraper.chub.lorebooks.insert import insert_chub_lorebook
from scraper.database.connection import Database, CursorFromConnectionFromPool
from scraper.chub.types.chub_lorebook import ChubLorebook, normalize_chub_lore_node
from scraper.paths import create_directory
from scraper.chub.strings import split_full_path
from iptr.jayson import load_json_with_err

"""
This script imports chub.ai users into PostgreSQL. IT ERASES ALL EXISTING DATA!!!
"""

root_path = Path('/srv/chub-archive/previous-archive/chub.ai/lorebooks')
create_directory(root_path)
root_path_images = Path('/srv/chub-archive/archive/hashed-data')
create_directory(root_path_images)
default_avatar = Path('../scraper/chub/lorebooks/default-lorebook-avatar.png').expanduser().resolve().absolute()


def process_lorebook(book_path: Path):
    chub_node_json_file = list(book_path.rglob('*.node.json'))
    if not len(chub_node_json_file):
        return

    chub_node_json = load_json_with_err(chub_node_json_file[0])
    if not chub_node_json.get('related_characters'):
        chub_node_json['related_characters'] = []
    if not chub_node_json.get('related_prompts'):
        chub_node_json['related_prompts'] = []
    if not chub_node_json.get('related_lorebooks'):
        chub_node_json['related_lorebooks'] = []
    chub_node_json['author'] = split_full_path(chub_node_json['fullPath'])[0]

    chub_node_ratings_json_file = list(book_path.rglob('*.ratings.json'))
    if len(chub_node_ratings_json_file):
        ratings = load_json_with_err(chub_node_ratings_json_file[0])
        if not ratings.get('errors'):
            chub_node_json['ratings'] = ratings

    chub_node = normalize_chub_lore_node(chub_node_json)
    del chub_node_json

    png_bytes = None
    if chub_node.avatar_url:
        try:
            r = requests.get(chub_node.avatar_url + f'?size=0.{random.randrange(1, 10 ** 16):03}', proxies={'http': 'http://172.0.4.7:9000', 'https': 'http://172.0.4.7:9000'})
            r.raise_for_status()
            if r.status_code == 200 and len(r.content) and r.headers['Content-Type'] in ['image/png', 'image/webp']:
                png_bytes = r.content
                Image.open(io.BytesIO(png_bytes))  # test image
        except:
            pass
    if not png_bytes:
        png_bytes = default_avatar.read_bytes()

    primary_format = chub_node.primaryFormat.lower()
    lorebook_files = []
    if primary_format == 'sillytavern':
        lorebook_files = list(book_path.rglob('*.lorebook_sillytavern.json'))
    elif primary_format == 'AGNAI':
        lorebook_files = list(book_path.rglob('*.lorebook_agnai.json'))
    if not len(lorebook_files):
        return
    lorebook_file = lorebook_files[0]
    if not lorebook_file.is_file():
        return

    lorebook_raw = lorebook_file.read_text()
    try:
        lorebook_data = json.loads(lorebook_raw)
    except JSONDecodeError:
        return

    try:
        lorebook = normalize_lorebook(lorebook_data, CHUB_TYPE_STR)
    except (ValidationError, ValueError):
        # Some lorebooks are fucked.
        return

    chub_lorebook = ChubLorebook(
        node=chub_node,
        lorebook=lorebook,
        png_bytes=png_bytes
    )
    insert_chub_lorebook(chub_lorebook, lorebook_raw, root_path_images)


Database.initialise(minconn=1, maxconn=100, database="chub_archive", user="chub_archive", password="eec3naigeeWu4yi0tohbeeB1eek4noom", host="172.0.3.109", port=5432)

with CursorFromConnectionFromPool() as cursor:
    cursor.execute('''
    truncate table chub_lorebook;
    truncate table chub_lorebook_def;
    ''')

files = sorted(list(root_path.iterdir()))

loading_bar = tqdm(total=len(files), desc='Importing lorebooks', smoothing=0.05, leave=True)
for book_path in files:
    loading_bar.write(str(book_path))
    process_lorebook(book_path)
    loading_bar.update(1)
