import json
import os
import re
import sys
import tempfile
import threading
from concurrent.futures import ThreadPoolExecutor, as_completed
from datetime import datetime, timezone
from io import BytesIO
from pathlib import Path

from PIL import Image
from tqdm import tqdm

from scraper.card_types.ccv2.infer import infer_specv2_card

script_dir = os.path.dirname(os.path.realpath(__file__))
sys.path.insert(0, str(Path(script_dir).parent))

from scraper.chub.types.node import clean_chub_node
from scraper.chub.chub import CHUB_TYPE_STR
from scraper.image import parse_card_metadata
from scraper.paths import create_directory
from scraper.chub.types.chub_character import ChubCharacter, ChubCharNode
from scraper.chub.types.chats_ratings import ChubChats, ChubRatings, normalize_chub_ratings
from iptr.jayson import load_json_with_err
from scraper.database.connection import Database, CursorFromConnectionFromPool
from scraper.chub.chars.insert import insert_chub_character

"""
This script imports chub.ai characters into PostgreSQL. IT ERASES ALL EXISTING DATA!!!

v1 cards are converted to v2.
"""

root_path = Path('/srv/chub-archive/previous-archive/chub.ai/characters')
create_directory(root_path)
root_path_images = Path('/srv/chub-archive/archive/hashed-data')
create_directory(root_path_images)

TIMESTAMP = datetime.now(timezone.utc)
PARENT_PID = os.getpid()


def process_card(card_path: Path):
    card_path = card_path.parent
    node_json_files = list(card_path.rglob('*.node.json'))
    card_raw_file = list(card_path.rglob('*.card_raw.json'))
    if len(node_json_files) and len(card_raw_file):
        node_data = load_json_with_err(node_json_files[0])

        for i in range(len(node_data['labels'])):
            try:
                node_data['labels'][i]['description'] = json.loads(node_data['labels'][i]['description'])
            except:
                pass

        node = ChubCharNode(**clean_chub_node(node_data))
        card_raw = card_raw_file[0].read_text()
        card_data = json.loads(card_raw)

        try:
            card_parsed = infer_specv2_card(card_data, node.author, CHUB_TYPE_STR)
        except ValueError:
            return

        chats_file = list(card_path.rglob('*.chats.json'))
        chats = ChubChats()
        if len(chats_file):
            chats_data = load_json_with_err(chats_file[0])
            if chats_data.get('chats'):  # file may be corrupted
                chats = ChubChats(chats=chats_data['chats'])

        ratings_file = list(card_path.rglob('*.ratings.json'))
        ratings = ChubRatings()
        if len(ratings_file):
            ratings_data = json.loads(ratings_file[0].read_text())
            if ratings_data.get('ratings'):  # file may be corrupted
                ratings = ChubRatings(ratings=normalize_chub_ratings(ratings_data['ratings']), enabled=ratings_data.get('enabled', True))

        # Find a valid card avatar and convert it to PNG if nessesary.
        png_file = list(card_path.rglob('*.card.png'))
        png_file_api = list(card_path.rglob('*.card_api.png')) + list(card_path.rglob('*.tavern_raw.png'))
        webp_file = list(card_path.rglob('*.card.webp'))
        webp_file_api = list(card_path.rglob('*.card_api.webp')) + list(card_path.rglob('*.tavern_raw.webp'))
        if len(png_file):
            png_file = png_file[0]
        elif len(png_file_api):
            png_file = png_file_api[0]
        elif len(webp_file):
            img = Image.open(BytesIO(webp_file[0].read_bytes())).convert('RGB')
            _, tmp_filename = tempfile.mkstemp()
            img.save(tmp_filename, format='PNG')
            png_file = Path(tmp_filename)
        elif len(webp_file_api):
            img = Image.open(BytesIO(webp_file_api[0].read_bytes())).convert('RGB')
            _, tmp_filename = tempfile.mkstemp()
            img.save(tmp_filename, format='PNG')
            png_file = Path(tmp_filename)
        else:
            tqdm.write(f'No png file: {card_path}')
            return

        chub_character = ChubCharacter(
            node=node,
            card_data=card_parsed,
            png_bytes=png_file.read_bytes(),
            chats=chats,
            ratings=ratings
        )
        insert_chub_character(chub_character, card_raw, root_path_images)
    else:
        def load(p: Path):
            card_r, card_p, err = parse_card_metadata(p, CHUB_TYPE_STR)
            if not card_r:
                return None, None
            return card_r, card_p,

        card_png_file = list(card_path.glob('*.card.png'))
        card_raw = png_file = None
        if card_png_file:
            png_file = card_png_file[0]
            card_raw, _ = load(png_file)
        if not card_raw:
            card_api_png_file = list(card_path.rglob('*.card_api.png'))
            if card_api_png_file:
                png_file = card_api_png_file[0]
                card_raw, _ = load(png_file)
        if not card_raw:
            card_api_webp_file = list(card_path.rglob('*.card.webp'))
            if card_api_webp_file:
                png_file = card_api_webp_file[0]
                card_raw, _ = load(png_file)
        if not card_raw or not png_file:
            return

        try:
            node_data = clean_chub_node(json.loads(card_raw))
            node = ChubCharNode(
                name=node_data['name'],
                author=node_data['extensions']['chub']['full_path'].split('/')[0],
                fullPath=node_data['extensions']['chub']['full_path'],
                id=node_data['extensions']['chub']['id'],
            )
            chub_character = ChubCharacter(
                node=node,
                card_data=infer_specv2_card(node_data, node.author, CHUB_TYPE_STR),
                png_bytes=png_file.read_bytes(),
            )
        except:
            return
    insert_chub_character(chub_character, card_raw, root_path_images)


def parse_item_name(n: str):
    parts = n.split(' - ')
    if len(parts) > 3:
        copy = parts.copy()
        del copy[0]
        del copy[-1]
        node_id_int = int(re.sub(r' *- *', '', parts[-1]))
        return parts[0], ' - '.join(copy).strip(' '), node_id_int
    node_id_int = int(re.sub(r' *- *', '', parts[2]))
    return parts[0], parts[1], node_id_int


Database.initialise(minconn=1, maxconn=100, database="chub_archive", user="chub_archive", password="eec3naigeeWu4yi0tohbeeB1eek4noom", host="172.0.3.109", port=5432)

with CursorFromConnectionFromPool() as cursor:
    cursor.execute(
        """
        truncate table chub_character;
        truncate table chub_character_def;
        """
    )

lock = threading.Lock()
seen_card_ids = set()
files = sorted(list(root_path.iterdir()))
loading_bar = tqdm(total=len(files), desc='Importing cards', smoothing=0.05, leave=True)


def process_file(author):
    found = set(list(author.rglob('*.card.png')) + list(author.rglob('*.card.json')))
    for item in found:
        base = str(item.name).rstrip('.card.png').rstrip('.card.json')
        try:
            author, name, card_id = parse_item_name(base)
        except:
            raise Exception(item)
        if card_id not in seen_card_ids:
            with lock:
                loading_bar.write(str(item))
                seen_card_ids.add(card_id)
            process_card(item)


with ThreadPoolExecutor() as executor:
    futures = {executor.submit(process_file, author) for author in files}
    for future in as_completed(futures):
        loading_bar.update(1)
