#!/usr/bin/env python3
import argparse
import logging
import os
import signal
import traceback
from datetime import datetime
from pathlib import Path

import scraper
from scraper import log
from scraper.catbox.process import catbox_download_pngs
from scraper.config.load import load_cfg
from scraper.database.connection import Database
from scraper.generic.mlpchag import load_mlpchag
from scraper.generic.pony_google_sheets import load_pony_sheet
from scraper.helpers import is_service_running
from scraper.paths import create_directory
from scraper.suicide import signal_handler, watchdog_expired, watchdog_suicide

signal.signal(signal.SIGINT, signal_handler)

_logger: logging.Logger

_SCRIPT_DIR = os.path.dirname(os.path.realpath(__file__))
_START_TIMESTAMP = datetime.now()


# TODO: make sure --exit-when-service works


def main(args):
    global _logger
    if args.debug:
        log_level = logging.DEBUG
    else:
        log_level = logging.INFO
    log.root_logger.init(log_level)
    _logger = log.root_logger.logger

    if args.exit_when_service:
        if is_service_running(args.exit_when_service):
            _logger.error(f'Service {args.exit_when_service} is running, exiting...')
            return
        else:
            _logger.info(f'Service {args.exit_when_service} is not running, continuing...')

    cfg = load_cfg(args)
    Database.initialise(minconn=1, maxconn=100, host=cfg.database.host, database=cfg.database.database, user=cfg.database.user, password=cfg.database.password)
    hashed_data_path = create_directory(cfg.hashed_data_path)
    catbox_threads = cfg.catbox.catbox_threads

    if args.target:
        _logger.info(f'Scraping targeted link...')
        targeted_new_cards, targeted_new_chub = catbox_download_pngs(catbox_threads, hashed_data_path, [args.target])
        _logger.info(f'Targeted -- New: {targeted_new_cards}. Found Chub: {targeted_new_chub}.')
        return

    pony_sheet_catbox = load_pony_sheet()
    _logger.info(f'Scraping {len(pony_sheet_catbox)} Catbox links from the pony sheet...')
    pony_sheet_new_cards, pony_sheet_new_chub = catbox_download_pngs(catbox_threads, hashed_data_path, pony_sheet_catbox, extra_metadata={'sourcedFrom': 'https://docs.google.com/spreadsheets/d/1J7BeqJVDS51cXF8Pgm2YZaFq-Z6ykSJT'})
    _logger.info(f'Pony sheet -- New: {pony_sheet_new_cards}. Found Chub: {pony_sheet_new_chub}.')

    mlpchag_cards = load_mlpchag()
    _logger.info(f'Scraping {len(mlpchag_cards)} Catbox links from mlpchag...')
    mlpchag_new_cards, mlpchag_new_chub = catbox_download_pngs(catbox_threads, hashed_data_path, mlpchag_cards, allow_non_catbox=True, specific_source='mlpchag.neocities.org')
    _logger.info(f'Ponydex -- New: {mlpchag_new_cards}. Found Chub: {mlpchag_new_chub}.')


if __name__ == '__main__':
    parser = argparse.ArgumentParser(description='')
    parser.add_argument('--config', default=Path(_SCRIPT_DIR, 'config', 'chub.yml'), help='Path to the config file. Default: config/chub.yml in the config directory next to the script.')
    parser.add_argument('--pages', type=int, default=205, help='How many pages to parse on desuarchive.')
    parser.add_argument('-d', '--debug', action='store_true', help='Enable debug logging.')
    parser.add_argument('--log-requests', action='store_true', help='Log all HTTP requests when debugging is enabled.')
    parser.add_argument('--run-time-limit', type=int, default=3600, help='If the program runs longer than this, kill it. This helps prevent cases where the scraper gets stuck. Value in seconds. Default: 3600 (one hour)')
    parser.add_argument('--exit-when-service', type=str, default=None, help='If this systemctl service is running, exit. Checks services running in --user mode.')
    parser.add_argument('--target', type=str, default=None, help='Download a card from the supplied URL.')
    args = parser.parse_args()

    signal.signal(signal.SIGALRM, watchdog_expired)
    bad = False
    try:
        main(args)
    except:
        traceback.print_exc()
        bad = True
    scraper.http_queue.quit()
    if bad:
        watchdog_suicide()
