#!/usr/bin/env python3
import argparse
import json
import zlib

from psycopg2.extras import RealDictCursor
from tqdm import tqdm

from scraper.card_types.infer import infer_anyspec_card
from scraper.card_types.item_extensions import parse_date_field
from scraper.card_types.st_timestamp import st_timestamp_to_datetime, parse_st_create_date_timestamp
from scraper.database.connection import Database, CursorFromConnectionFromPool


def process_table(rows, table_name, primary_key: tuple):
    for row in tqdm(rows, desc=table_name):
        raw_card = json.loads(zlib.decompress(row['raw']).decode())
        card_def = row['definition']
        if raw_card.get('spec_version') != '3.0':
            if raw_card.get('spec_version') is not None:
                tqdm.write(f'Skipping {raw_card.get("spec_version")}')
            continue

        try:
            card = infer_anyspec_card(card_def, source=row['source'])

            # Fix creation date.
            if raw_card.get('create_date'):
                card.data.creation_date = int(parse_st_create_date_timestamp(raw_card['create_date']).timestamp())
            else:
                card.data.creation_date = raw_card['data'].get('creation_date')
            card.data.modification_date = raw_card['data'].get('modification_date')

            # Update the char archive extension to the latest type.
            card.data.extensions.char_archive.created = parse_date_field(card.data.creation_date)
            card.data.extensions.char_archive.added = row['added']

            # Put the source ID in the char archive extension.
            card.data.extensions.char_archive.source_id = row['id']
        except:
            print(row['increment'], json.dumps(raw_card))
            raise

        with CursorFromConnectionFromPool(cursor_factory=RealDictCursor) as cursor:
            cursor.execute(f"UPDATE {table_name} SET definition = %s WHERE {' AND '.join([f'{k} = %s' for k in primary_key])}", (card.model_dump_json(), *[row[k] for k in primary_key]))
        tqdm.write(f'{row[primary_key[0]]} -- {card.data.name} -- {card.data.creator}')


def main(args):
    Database.initialise(minconn=1, maxconn=100, host='172.0.3.101', database='char_archive', user='char_archive', password='hei3ucheet5oochohjongeisahV3mei0')
    with CursorFromConnectionFromPool(cursor_factory=RealDictCursor) as cursor:
        cursor.execute("SELECT * FROM risuai_character_def ORDER BY added desc" + (' LIMIT 20' if args.test else ''))
        process_table(cursor.fetchall(), 'risuai_character_def', ('increment',))


if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument('-t', '--test', action='store_true')
    args = parser.parse_args()
    main(args)
