#!/usr/bin/env python3
import argparse
import json
import zlib
from typing import Union

from psycopg2.extras import RealDictCursor
from tqdm import tqdm

from scraper.card_types.ccv2.character import TavernCardv2
from scraper.card_types.ccv3.character import CharacterCardV3
from scraper.card_types.infer import infer_anyspec_card
from scraper.card_types.st_timestamp import parse_st_create_date_timestamp
from scraper.database.connection import Database, CursorFromConnectionFromPool


def process_table(rows, table_name, primary_key: tuple):
    for row in tqdm(rows, desc=table_name):
        card: Union[TavernCardv2, CharacterCardV3]
        raw_card = json.loads(zlib.decompress(row['raw']).decode())
        card_def = row['definition']

        try:
            card = infer_anyspec_card(card_def, source=row['source'])

            if raw_card.get('data', {}).get('creator') is not None and raw_card.get('data', {})['creator'] != '':
                # Fix author.
                card.data.creator = raw_card['data']['creator']

            # Fix creation date.
            if isinstance(card, TavernCardv2):
                card.create_date = parse_st_create_date_timestamp(raw_card.get('create_date'))
            elif isinstance(card, CharacterCardV3):
                card.data.creation_date = raw_card['data']['creation_date']
                card.data.modification_date = raw_card['data']['modification_date']

            # Update the char archive extension to the latest type.
            if isinstance(card, TavernCardv2):
                card.data.extensions.char_archive.created = card.create_date
            elif isinstance(card, CharacterCardV3):
                card.data.extensions.char_archive.created = card.data.creation_date

            card.data.extensions.char_archive.added = row['added']

            # Remove unnessesary data from metadata.
            if row['metadata'].get('created'):
                del row['metadata']['created']
            if row['metadata'].get('source_url'):
                del row['metadata']['source_url']
        except:
            print(row['card_data_hash'], json.dumps(raw_card))
            raise

        with CursorFromConnectionFromPool(cursor_factory=RealDictCursor) as cursor:
            cursor.execute(f"UPDATE {table_name} SET definition = %s , metadata = %s WHERE {' AND '.join([f'{k} = %s' for k in primary_key])}", (card.model_dump_json(), json.dumps(row['metadata']), *[row[k] for k in primary_key]))
        tqdm.write(f'{row[primary_key[0]]} -- {card.data.name} -- {card.data.creator}')


def main(args):
    Database.initialise(minconn=1, maxconn=100, host='172.0.3.101', database='char_archive', user='char_archive', password='hei3ucheet5oochohjongeisahV3mei0')
    with CursorFromConnectionFromPool(cursor_factory=RealDictCursor) as cursor:
        print('Fetching generic')
        cursor.execute("SELECT * FROM generic_character_def ORDER BY added desc" + (' LIMIT 20' if args.test else ''))
        process_table(cursor.fetchall(), 'generic_character_def', ('card_data_hash', 'increment'))


if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument('-t', '--test', action='store_true')
    args = parser.parse_args()
    main(args)
