#!/usr/bin/env python3
import argparse
import logging
import os
import signal
import traceback
from datetime import datetime
from pathlib import Path

import scraper
from scraper import log
from scraper.character_tavern.execute_get_auth_header import get_character_tavern_auth, test_character_tavern_auth
from scraper.character_tavern.process import process_char_tavern_nodes
from scraper.character_tavern.search import fetch_character_tavern_sorted
from scraper.character_tavern.users import process_char_tavern_users
from scraper.config.load import load_cfg
from scraper.database.connection import Database
from scraper.helpers import valid_date
from scraper.paths import create_directory
from scraper.suicide import signal_handler, watchdog_expired, watchdog_suicide
from scraper.time import calculate_elapsed_time

signal.signal(signal.SIGINT, signal_handler)

_logger: logging.Logger

_SCRIPT_DIR = os.path.dirname(os.path.realpath(__file__))
_START_TIMESTAMP = datetime.now()


# TODO: add 'since' arg like chub

def main(args):
    global _logger
    if args.debug:
        log_level = logging.DEBUG
    else:
        log_level = logging.INFO
    log.root_logger.init(log_level)
    _logger = log.root_logger.logger

    cfg = load_cfg(args)
    Database.initialise(minconn=1, maxconn=100, host=cfg.database.host, database=cfg.database.database, user=cfg.database.user, password=cfg.database.password)
    create_directory(cfg.hashed_data_path)
    chartavern_threads = cfg.character_tavern.http_workers

    if args.test:
        _logger.warning('Test mode enabled!')

    if args.since:
        _logger.info(f'Scraping nodes updated after {args.since.astimezone()}')

    _logger.info("Getting authorization header...")
    auth_header = get_character_tavern_auth()

    if auth_header:
        test_status_code = test_character_tavern_auth(auth_header)
        if test_status_code != 200:
            _logger.fatal(f"Scraped authorization header was invalid, got code {test_status_code} when testing")
            watchdog_suicide()
        _logger.info(f'Successfully obtained and tested authorization header: "{auth_header}"')
    else:
        _logger.fatal("Failed to get authorization header")
        watchdog_suicide()

    _logger.info('Fetching newest characters...')
    characters = fetch_character_tavern_sorted('asc', thread_count=chartavern_threads, test_mode=args.test, auth_header=auth_header, since_date=args.since)

    users = set()
    for character in characters:
        users.add(character.path.split('/')[0])

    _logger.info(f'Got {len(characters)} characters')
    process_char_tavern_nodes(characters, chartavern_threads)

    _logger.info(f'Scraping {len(users)} users')
    users_start = datetime.now()
    new_users = process_char_tavern_users(users, chartavern_threads)
    _logger.info(f'Got {new_users} new users')

    user_time_str = calculate_elapsed_time(users_start)
    _logger.info(f'-- Users scraping completed in {user_time_str} --'.upper())

    _logger.info('-- DONE --')
    total_time_str = calculate_elapsed_time(_START_TIMESTAMP)
    _logger.info(f'Scraping took {total_time_str}')
    _logger.info(f'CHARACTER TAVERN STATUS -- Characters: {len(characters)}, Users: {len(users)}')


if __name__ == '__main__':
    parser = argparse.ArgumentParser(description='')
    parser.add_argument('--config', default=Path(_SCRIPT_DIR, 'config', 'chub.yml'), help='Path to the config file. Default: config/chub.yml in the config directory next to the script.')
    parser.add_argument('-d', '--debug', action='store_true', help='Enable debug logging.')
    parser.add_argument('--log-requests', action='store_true', help='Log all HTTP requests when debugging is enabled.')
    parser.add_argument('--run-time-limit', type=int, default=10800, help='If the program runs longer than this, kill it. This helps prevent cases where the scraper gets stuck. Value in seconds. Default: 10800 (3 hours)')
    parser.add_argument('--test', action='store_true', help='Test mode.')
    parser.add_argument('--since', help='Scrape all cards since this timedelta. Examples: "5 minutes ago", "one week ago", "25 hours", "3 sec"', type=valid_date)
    args = parser.parse_args()

    signal.signal(signal.SIGALRM, watchdog_expired)
    bad = False
    try:
        main(args)
    except:
        traceback.print_exc()
        bad = True
    scraper.http_queue.quit()
    if bad:
        watchdog_suicide()
