import time
import warnings
from abc import ABC
from copy import deepcopy
from typing import Optional, Union
import transformers.sentence_detect

import torch

from .file_utils import add_start_docstrings

STOPPING_CRITERIA_INPUTS_DOCSTRING = r"""
    Args:
        input_ids (:obj:`torch.LongTensor` of shape :obj:`(batch_size, sequence_length)`):
            Indices of input sequence tokens in the vocabulary.

            Indices can be obtained using :class:`~transformers.BertTokenizer`. See
            :meth:`transformers.PreTrainedTokenizer.encode` and :meth:`transformers.PreTrainedTokenizer.__call__` for
            details.

            `What are input IDs? <../glossary.html#input-ids>`__
        scores (:obj:`torch.FloatTensor` of shape :obj:`(batch_size, config.vocab_size)`):
            Prediction scores of a language modeling head. These can be scores for each vocabulary token before SoftMax
            or scores for each vocabulary token after SoftMax.
        kwargs:
            Additional stopping criteria specific kwargs.

    Return:
        :obj:`bool`. :obj:`False` indicates we should continue, :obj:`True` indicates we should stop.

"""


class StoppingCriteria(ABC):
    """Abstract base class for all stopping criteria that can be applied during generation."""

    @add_start_docstrings(STOPPING_CRITERIA_INPUTS_DOCSTRING)
    def __call__(self, input_ids: torch.LongTensor, score: torch.FloatTensor,
                 **kwargs) -> bool:
        raise NotImplementedError("StoppingCriteria needs to be subclassed")


class MaxLengthCriteria(StoppingCriteria):
    """
    This class can be used to stop generation whenever the full generated number of tokens exceeds :obj:`max_length`.
    Keep in mind for decoder-only type of transformers, this will include the initial prompted tokens.

    Args:
        max_length (:obj:`int`):
            The maximum length that the output sequence can have in number of tokens.
    """

    def __init__(self, max_length: int):
        self.max_length = max_length

    @add_start_docstrings(STOPPING_CRITERIA_INPUTS_DOCSTRING)
    def __call__(self, input_ids: torch.LongTensor, scores: torch.FloatTensor,
                 **kwargs) -> bool:
        return input_ids.shape[-1] >= self.max_length


class MinLengthCriteria(StoppingCriteria):
    """
    This is a placeholder class that holds :obj:`min_length`, the actual
    evaluation is in the `obj:`StoppingCriteria` class which uses it to
    short circuit any evaluation.
    """

    def __init__(self, min_length: int):
        self.min_length = min_length

    def __call__(self, input_ids: torch.LongTensor, scores: torch.FloatTensor,
                 **kwargs) -> bool:
        return False


class MaxTimeCriteria(StoppingCriteria):
    """
    This class can be used to stop generation whenever the full generation exceeds some amount of time. By default, the
    time will start being counted when you initialize this function. You can override this by passing an
    :obj:`initial_time`.

    Args:
        max_time (:obj:`float`):
            The maximum allowed time in seconds for the generation.
        initial_time (:obj:`float`, `optional`, defaults to :obj:`time.time()`):
            The start of the generation allowed time.
    """

    def __init__(self, max_time: float,
                 initial_timestamp: Optional[float] = None):
        self.max_time = max_time
        self.initial_timestamp = time.time() if initial_timestamp is None else initial_timestamp

    @add_start_docstrings(STOPPING_CRITERIA_INPUTS_DOCSTRING)
    def __call__(self, input_ids: torch.LongTensor, scores: torch.FloatTensor,
                 **kwargs) -> bool:
        return time.time() - self.initial_timestamp > self.max_time


class SentenceStoppingCriteria(StoppingCriteria):
    def __init__(self, tokenizer, start_idx: int = 0):
        self.tokenizer = tokenizer
        self.start_idx = start_idx

    def __call__(self, input_ids: torch.LongTensor, scores: torch.FloatTensor,
                 **kwargs) -> bool:
        return transformers.sentence_detect.is_sentence_tokens(
            self.tokenizer, input_ids[0, self.start_idx:].tolist())


class SequenceStoppingCriteria(StoppingCriteria):
    def __init__(self, sequence: torch.LongTensor):
        self.sequence = sequence

    def __call__(self, input_ids: torch.LongTensor, scores: torch.FloatTensor,
                 **kwargs) -> bool:
        return input_ids.shape[1] > self.sequence.shape[1] and \
               torch.all(
                   input_ids[:, -self.sequence.shape[1]:] == self.sequence,
                   dim=1)


class StoppingCriteriaList(list):
    @add_start_docstrings(STOPPING_CRITERIA_INPUTS_DOCSTRING)
    def __call__(self, input_ids: torch.LongTensor, scores: torch.FloatTensor,
                 **kwargs) -> Union[bool, str]:
        if self.min_length > input_ids.shape[-1]:
            return False

        for criteria in self:
            if criteria(input_ids, scores):
                return type(criteria).__name__
        return False

    @property
    def min_length(self) -> Optional[int]:
        for stopping_criterium in self:
            if isinstance(stopping_criterium, MinLengthCriteria):
                return stopping_criterium.min_length
        return 0

    @property
    def max_length(self) -> Optional[int]:
        for stopping_criterium in self:
            if isinstance(stopping_criterium, MaxLengthCriteria):
                return stopping_criterium.max_length
        return None


def validate_stopping_criteria(stopping_criteria: StoppingCriteriaList,
                               max_length: int) -> StoppingCriteriaList:
    stopping_max_length = stopping_criteria.max_length
    new_stopping_criteria = deepcopy(stopping_criteria)
    if stopping_max_length is not None and stopping_max_length != max_length:
        warnings.warn(
            "You set different `max_length` for stopping criteria and `max_length` parameter",
            UserWarning)
    elif stopping_max_length is None:
        new_stopping_criteria.append(MaxLengthCriteria(max_length=max_length))
    return new_stopping_criteria
