import { equals } from 'rambda'
import { ChangeMap, HistoryRoot, HistoryStateId, HistoryStepType } from './history'
import { cloneSection, Section, SectionDiff, SectionHandle, SectionId, sectionsEqual } from './section'
import { documentToJsonReplacer } from './util'

/**
 * Map from section id to section.
 */
type SectionMap = Map<SectionId, Section>

/**
 * Root datastructure of the document.
 */
export class Document {
    private sections: SectionMap
    private order: Array<SectionId>
    private history: HistoryRoot
    private dirtySections: ChangeMap
    private step: number

    constructor() {
        this.sections = new Map()
        this.order = []
        this.history = new HistoryRoot()
        this.dirtySections = new Map()
        this.step = 0
    }

    private addToOrder(sectionId: SectionId, after?: SectionId) {
        const afterIndex = after ? this.order.indexOf(after) : -1
        this.order.splice(afterIndex === -1 ? this.order.length : afterIndex + 1, 0, sectionId)
    }
    private removeFromOrder(sectionId: SectionId) {
        this.order = this.order.filter((id) => id !== sectionId)
    }

    /**
     * Get all sections in the current history state ordered by the section order.
     * @returns All active sections ordered by the section order.
     */
    getSections() {
        return this.order
            .map((id) => ({ id, section: this.sections.get(id) }))
            .filter((section) => Boolean(section.section)) as Array<SectionHandle>
    }

    /**
     * Get a section in the current history state.
     * @param id Section id corresponding to the section.
     * @returns Section corresponding to `id`.
     */
    getSection(id: SectionId) {
        return this.sections.get(id)
    }

    /**
     * Get the sections that come before another section in the current history state ordered by the section order.
     * @param id Section id to search in the order.
     * @param count Maximum amount of sections to return.
     * @returns `count` active sections that come before `id` in the section order.
     */
    getSectionsBefore(id: SectionId, count: number) {
        const orderIndex = this.order.indexOf(id)
        if (!orderIndex) return [] as Array<SectionHandle>
        return this.order
            .slice(Math.max(0, orderIndex - count), orderIndex)
            .map((id) => ({ id, section: this.sections.get(id) }))
            .filter((section) => Boolean(section.section)) as Array<SectionHandle>
    }

    /**
     * Get the history state ids of all descendents of the current state in the history tree.
     * @returns List of history state ids of all descendents.
     */
    getDescendents() {
        return this.history.getCurrentNode().children
    }

    /**
     * Check if any active changes are uncommitted.
     * @returns Boolean marking if any active changes are uncommitted.
     */
    isDirty() {
        return this.dirtySections.size > 0
    }

    /**
     * Push new changes into the set of active changes.
     * @param change Map of sections that should be changed.
     */
    pushChange(change: Map<SectionId, { changedSection?: Section; after?: SectionId }>) {
        if (change.size === 0) return
        // if this is the first change since last history push or we're in the root history node,
        // create a new history state
        if (
            this.dirtySections.size === 0 &&
            (this.history.getCurrentChangeSet().size > 0 || this.history.isCurrentRoot())
        )
            this.history.pushState()
        console.log('changed nodes', change)
        // iterate all changed nodes and update sections
        for (const [sectionId, { changedSection, after }] of change.entries()) {
            if (this.sections.has(sectionId)) {
                // document already contains section
                if (changedSection) {
                    // update existing section if node exists
                    // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
                    const section = this.sections.get(sectionId)!
                    if (sectionsEqual(section, changedSection)) {
                        // if the node is equal, remove it from the changeset
                        this.dirtySections.delete(sectionId)
                    } else if (this.dirtySections.has(sectionId)) {
                        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
                        const step = this.dirtySections.get(sectionId)!
                        switch (step.type) {
                            case HistoryStepType.create: {
                                step.section = changedSection
                                break
                            }
                            case HistoryStepType.update: {
                                step.diff = new SectionDiff(section, changedSection)
                                break
                            }
                            case HistoryStepType.remove: {
                                this.dirtySections.set(sectionId, {
                                    type: HistoryStepType.update,
                                    diff: new SectionDiff(section, changedSection),
                                })
                                break
                            }
                        }
                    } else {
                        this.dirtySections.set(sectionId, {
                            type: HistoryStepType.update,
                            diff: new SectionDiff(section, changedSection),
                        })
                    }
                } else {
                    // remove section otherwise
                    this.dirtySections.set(sectionId, {
                        type: HistoryStepType.remove,
                        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
                        previous: this.sections.get(sectionId)!,
                        after: after,
                    })
                }
            } else if (this.dirtySections.has(sectionId)) {
                // section is new but already tracked in current changeset
                if (changedSection) {
                    // update changeset if node exists
                    // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
                    const step = this.dirtySections.get(sectionId)!
                    switch (step.type) {
                        case HistoryStepType.create: {
                            step.section = changedSection
                            break
                        }
                        case HistoryStepType.update: {
                            throw new Error("can't update non-existing section")
                        }
                        case HistoryStepType.remove: {
                            throw new Error("can't update non-existing section")
                        }
                    }
                } else {
                    // remove section from the changeset otherwise
                    this.dirtySections.delete(sectionId)
                }
            } else {
                // add the new section otherwise
                if (changedSection) {
                    // add a new node
                    if (after) {
                        // insert after previous node if there is one
                        this.dirtySections.set(sectionId, {
                            type: HistoryStepType.create,
                            section: changedSection,
                            after: after,
                        })
                    } else {
                        // append it at the end otherwise
                        this.dirtySections.set(sectionId, {
                            type: HistoryStepType.create,
                            section: changedSection,
                        })
                    }
                } else {
                    // node removed and not in current document so untrack it
                    this.dirtySections.delete(sectionId)
                }
            }
        }
    }

    /**
     * Check if active changes can be pushed into the history.
     * @returns Boolean marking if any changes can be committed.
     */
    canPushHistory() {
        return this.dirtySections.size > 0
    }

    /**
     * Commit active changes into the history.
     * @returns Boolean marking if any changes were committed.
     */
    pushHistory() {
        if (!this.canPushHistory()) {
            return false
        }
        console.log('push history')
        for (const [sectionId, step] of this.dirtySections.entries()) {
            switch (step.type) {
                case HistoryStepType.create: {
                    this.sections.set(sectionId, step.section)
                    this.addToOrder(sectionId, step.after)
                    break
                }
                case HistoryStepType.update: {
                    // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
                    this.sections.set(sectionId, step.diff.apply(this.sections.get(sectionId)!))
                    break
                }
                case HistoryStepType.remove: {
                    this.sections.delete(sectionId)
                    this.removeFromOrder(sectionId)
                    break
                }
            }
        }
        this.history.appendChanges(this.dirtySections)
        this.dirtySections.clear()
        this.step += 1
        return true
    }

    /**
     * Check if the current state can be popped into the parent state.
     * @returns Boolean marking if the current state can be popped.
     */
    canPopHistory() {
        return this.dirtySections.size === 0 && this.history.getCurrentNode().parent
    }

    /**
     * Pop the current state and return to the parent state in the history tree.
     * @returns List of changes that were applied between the previous and the new state.
     */
    popHistory() {
        if (!this.canPopHistory()) {
            return
        }
        const changeSet = this.history.getCurrentChangeSet()
        if (!this.history.popState()) {
            return
        }
        if (!changeSet) return
        for (const [sectionId, step] of changeSet.entries()) {
            switch (step.type) {
                case HistoryStepType.create: {
                    this.sections.delete(sectionId)
                    this.removeFromOrder(sectionId)
                    break
                }
                case HistoryStepType.update: {
                    // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
                    this.sections.set(sectionId, step.diff.undo(this.sections.get(sectionId)!))
                    break
                }
                case HistoryStepType.remove: {
                    this.sections.set(sectionId, cloneSection(step.previous))
                    this.addToOrder(sectionId, step.after)
                    break
                }
            }
        }
        this.step -= 1
        return changeSet
    }

    /**
     * Check if a child state can be descended into.
     * @returns Boolean marking if a child state can be descended into.
     */
    canDescendHistory() {
        const descendents = this.history.getCurrentNode().children
        return descendents && descendents.size > 0
    }

    /**
     * Descend into a child state of the history tree.
     * @param branch History state id of the child state to be descended into.
     * @returns List of changes that were applied between the new and the previous state.
     */
    descendHistory(branch: HistoryStateId) {
        if (!this.canDescendHistory() || !this.history.descendState(branch)) {
            return false
        }
        const changeSet = this.history.getCurrentChangeSet()
        if (!changeSet) return
        for (const [sectionId, step] of changeSet.entries()) {
            switch (step.type) {
                case HistoryStepType.create: {
                    this.sections.set(sectionId, step.section)
                    this.addToOrder(sectionId, step.after)
                    break
                }
                case HistoryStepType.update: {
                    // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
                    this.sections.set(sectionId, step.diff.apply(this.sections.get(sectionId)!))
                    break
                }
                case HistoryStepType.remove: {
                    this.sections.delete(sectionId)
                    this.removeFromOrder(sectionId)
                    break
                }
            }
        }
        this.step += 1
        return changeSet
    }

    /**
     * @returns String representing the document.
     */
    toString() {
        return 'Document ' + JSON.stringify(this.toJSON(), documentToJsonReplacer, 4)
    }

    /**
     * @returns Object representing the document.
     */
    toJSON() {
        return {
            step: this.step,
            dirtySections: this.dirtySections,
            sections: this.sections,
            order: this.order,
            history: this.history,
        }
    }
}
