import {
    useRef,
    useImperativeHandle,
    forwardRef,
    CSSProperties,
    useState,
    MutableRefObject,
    useEffect,
    useCallback,
    useMemo,
    UIEvent,
} from 'react'
import { EditorView, EditorProps, DirectEditorProps } from 'prosemirror-view'
import { EditorState, Transaction } from 'prosemirror-state'
import { Schema } from 'prosemirror-model'
import { flushSync } from 'react-dom'

// eslint-disable-next-line @typescript-eslint/no-unused-vars
type Config<S extends Schema<string, string>> = Parameters<typeof EditorState.create>[0]
export interface ProseMirrorHandle {
    view: EditorView | null
    root: HTMLDivElement | null
}
interface PropsBase<S extends Schema<string, string>> extends EditorProps<unknown, S> {
    state: EditorState<S>
    style?: CSSProperties
    className?: string
    dispatchTransaction?: (transaction: Transaction<S>, state: EditorState<S>) => void
    handleScroll?: (view: EditorView, event: UIEvent<HTMLDivElement>) => void
}
type ProseMirrorProps<S extends Schema<string, string>> = PropsBase<S>
export type TransformState<S extends Schema<string, string>> = (
    transform: (state: EditorState<S>) => Transaction<S> | undefined
) => EditorState<S>

export const useProseMirror = <S extends Schema<string, string>>(
    config: Config<S>
): [EditorState<S>, TransformState<S>] => {
    const [state, setState] = useState(() => EditorState.create<S>(config as any) as EditorState<S>)
    const stateRef = useRef(state)
    const transformState = useCallback((transform) => {
        const internalTransformState = (tr?: Transaction) => {
            if (!tr) return stateRef
            const newState = stateRef.current.apply(tr)
            flushSync(() => {
                stateRef.current = newState
                setState(newState)
            })
            return newState
        }
        return internalTransformState(transform(stateRef.current))
    }, []) as TransformState<S>
    return [state, transformState]
}

export const ProseMirror = forwardRef<ProseMirrorHandle, ProseMirrorProps<Schema<string, string>>>(function ProseMirror(
    { state, dispatchTransaction, handleScroll, style, className, ...restProps },
    ref
): JSX.Element {
    const rootRef: MutableRefObject<HTMLDivElement | null> = useRef(null)
    const viewRef: MutableRefObject<EditorView<any> | null> = useRef(null)

    useImperativeHandle(ref, () => ({
        get view() {
            return viewRef.current
        },
        get root() {
            return rootRef.current
        },
    }))

    if (viewRef.current) {
        viewRef.current.updateState(state)
    }

    const dispatchTransactionRef = useRef(dispatchTransaction)
    dispatchTransactionRef.current = dispatchTransaction

    const handleScrollRef = useRef(handleScroll)
    handleScrollRef.current = handleScroll

    const editorProps = useMemo(
        (): Partial<DirectEditorProps> => ({
            ...restProps,
            dispatchTransaction: (transaction) => {
                if (!viewRef.current) {
                    return
                }
                dispatchTransactionRef.current
                    ? dispatchTransactionRef.current(transaction, viewRef.current.state)
                    : viewRef.current.updateState(viewRef.current.state.apply(transaction))
            },
        }),
        // eslint-disable-next-line react-hooks/exhaustive-deps
        []
    )

    useEffect(() => {
        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
        viewRef.current = new EditorView(rootRef.current!, {
            state,
            ...editorProps,
        } as DirectEditorProps)
        return () => {
            viewRef.current && viewRef.current.destroy()
        }
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [])

    const view = useMemo(
        (): JSX.Element => (
            <div
                ref={rootRef}
                style={style}
                className={className}
                onScroll={(event) =>
                    viewRef.current && handleScrollRef.current?.call(handleScrollRef.current, viewRef.current, event)
                }
            />
        ),
        [style, className]
    )
    return view
})
