import 'prosemirror-view/style/prosemirror.css'
import './prosemirror.css'

import { EditorState, Transaction } from 'prosemirror-state'
import { Node } from 'prosemirror-model'
import { CSSProperties, forwardRef, useCallback, useImperativeHandle, useRef } from 'react'
import { EditorView } from 'prosemirror-view'

import { Document } from '../../data/document/document'
import { SectionType } from '../../data/document/section'
import { UniqueId } from '../../data/document/util'

import { ProseMirrorHandle, ProseMirror, useProseMirror } from './prosemirror'
import { clipboardTransform, placeholder, propsOverride, nodeIds, devTool, keymap, inspectNodeChanges } from './plugins'
import { schema } from './schema'
import { editorKeymap } from './keymap'
import { applyMetaMarks, redoChanges, reloadDocument, undoChanges, updateOriginMarks } from './actions'
import { formattingToMark, nodeToSection, originToMark, SectionMetaFormatting, SectionMetaOrigin } from './glue'

export type EditorHandle = {
    view: EditorView | null
    root: HTMLDivElement | null
    document: Document
    generate: () => void
    undo: () => void
    redo: (branch: UniqueId) => void
    reload: () => void
    bolden: () => void
    italicize: () => void
}

export type EditorProps = {
    style?: CSSProperties
    className?: string
    onTransaction?: (transaction: Transaction) => void
    onDocumentChange?: (document: Document) => void
}

export const Editor = forwardRef<EditorHandle, EditorProps>(function Editor(
    { style, className, onTransaction, onDocumentChange },
    ref
) {
    const documentRef = useRef(new Document())
    const viewRef = useRef<ProseMirrorHandle>(null)

    const onTransactionRef = useRef(onTransaction)
    onTransactionRef.current = onTransaction

    const onDocumentChangeRef = useRef(onDocumentChange)
    onDocumentChangeRef.current = onDocumentChange

    const onNodesChanged = useCallback((nodes: Map<UniqueId, { node?: Node; after?: Node }>) => {
        console.log('changed nodes', nodes)
        const change = new Map()
        nodes.forEach(({ node, after }, id) => {
            change.set(id, { changedSection: node ? nodeToSection(id, node) : undefined, after: after?.attrs.id })
        })
        documentRef.current.pushChange(change)
        onDocumentChangeRef.current?.call(onDocumentChangeRef.current, documentRef.current)
    }, [])

    const [state, transformState] = useProseMirror<typeof schema>({
        schema: schema,
        plugins: [
            placeholder(),
            keymap(editorKeymap),
            clipboardTransform(),
            nodeIds(),
            propsOverride({
                handlePaste: () => false,
                editable: () => true,
                handleKeyDown: () => false,
                handleClick: (view, pos, event) => {
                    return true
                },
                handleDOMEvents: {
                    blur: (view, event) => {
                        console.log('blur')
                        documentRef.current.pushHistory()
                        onDocumentChangeRef.current?.call(onDocumentChangeRef.current, documentRef.current)
                        return true
                    },
                },
            }),
            inspectNodeChanges(onNodesChanged),
            devTool(),
        ],
    })

    const dispatchTransaction = useCallback(
        (tr: Transaction<any>, state: EditorState<any>) => {
            console.log('# dispatch')
            updateOriginMarks(tr, state)
            transformState(() => tr)
            onTransactionRef.current?.call(onTransactionRef.current, tr)
        },
        [transformState]
    )

    const generate = useCallback(() => {
        transformState((state) => {
            const tr = state.tr
            console.log('generate')
            const addedText = 'This is AI generated text!'
            tr.insert(tr.doc.content.size - 1, schema.text(addedText, [schema.mark(schema.marks.ai_text)]))
            return tr
        })
        documentRef.current.pushHistory()
        onDocumentChangeRef.current?.call(onDocumentChangeRef.current, documentRef.current)
    }, [transformState])

    const bolden = useCallback(() => {
        transformState((state) => {
            const tr = state.tr
            console.log('bolden')
            // NOTE: this will not cause a proper document change event (step map is empty), why?
            tr.addMark(tr.selection.from, tr.selection.to, schema.mark(schema.marks.bold))
            tr.setMeta('forceupdate', true)
            return tr
        })
        documentRef.current.pushHistory()
        onDocumentChangeRef.current?.call(onDocumentChangeRef.current, documentRef.current)
    }, [transformState])

    const italicize = useCallback(() => {
        transformState((state) => {
            const tr = state.tr
            console.log('italicize')
            // NOTE: this will not cause a proper document change event (step map is empty), why?
            tr.addMark(tr.selection.from, tr.selection.to, schema.mark(schema.marks.italic))
            tr.setMeta('forceupdate', true)
            return tr
        })
        documentRef.current.pushHistory()
        onDocumentChangeRef.current?.call(onDocumentChangeRef.current, documentRef.current)
    }, [transformState])

    const reload = useCallback(() => {
        console.log('== reload')
        const document = documentRef.current
        transformState((state) => {
            const tr = state.tr
            reloadDocument(tr, document)
            return tr
        })
    }, [transformState])

    const undo = useCallback(() => {
        const document = documentRef.current
        const changes = document.popHistory()
        console.log('<= changes', changes)
        if (!changes) return
        transformState((state) => {
            const tr = state.tr
            undoChanges(tr, changes, document)
            tr.setMeta('internal', true)
            return tr
        })
        onDocumentChangeRef.current?.call(onDocumentChangeRef.current, documentRef.current)
    }, [transformState])

    const redo = useCallback(
        (branch: UniqueId) => {
            const document = documentRef.current
            const changes = document.descendHistory(branch)
            console.log('=> changes', changes)
            if (!changes) return
            transformState((state) => {
                const tr = state.tr
                redoChanges(tr, changes, document)
                tr.setMeta('internal', true)
                return tr
            })
            onDocumentChangeRef.current?.call(onDocumentChangeRef.current, documentRef.current)
        },
        [transformState]
    )

    const unloadSections = useCallback(() => {
        if (!viewRef.current?.view) return
        const document = documentRef.current
        // don't unload when document is being edited
        if (document.isDirty()) return
        const docDom = viewRef.current.view.dom as HTMLElement
        const domParent = docDom.parentElement
        if (!domParent) return
        // only unload when scrolled to the bottom
        if (domParent.scrollTop + domParent.offsetHeight < domParent.scrollHeight - 5) return
        const domScroll = domParent.scrollTop || 0
        transformState((state) => {
            let deleteTo = 0
            state.doc.descendants((node, pos) => {
                if (!viewRef.current?.view) return false
                const nodeDom = viewRef.current.view.nodeDOM(pos) as HTMLElement | undefined
                if (!nodeDom) return false
                if (nodeDom.offsetTop + nodeDom.offsetHeight < domScroll - 1000) {
                    deleteTo = pos + node.nodeSize
                }
                return false
            })
            if (deleteTo === 0) return
            const tr = state.tr
            try {
                tr.delete(0, deleteTo)
            } catch (error: unknown) {
                console.error(error)
            }
            tr.setMeta('internal', true)
            return tr
        })
    }, [transformState])

    const reloadSections = useCallback(() => {
        if (!viewRef.current?.view) return
        const docDom = viewRef.current.view.dom as HTMLElement
        const docScroll = docDom.parentElement?.scrollTop || 0
        // only load when scrolled to the top
        if (docScroll >= 500) return
        const document = documentRef.current
        transformState((state) => {
            const firstNode = state.doc.firstChild
            if (!firstNode || !firstNode.attrs.id) {
                // reload when no section is loaded
                reload()
                return
            }
            const sectionsBefore = document.getSectionsBefore(firstNode.attrs.id, 25)
            if (sectionsBefore.length === 0) return
            const tr = state.tr
            for (const { id, section } of sectionsBefore.reverse()) {
                if (section.type !== SectionType.text) {
                    // skip non-text nodes for now
                    continue
                }
                const paragraph = schema.nodes.paragraph.create(
                    { id },
                    section.text ? schema.text(section.text) : undefined
                )
                tr.insert(0, paragraph)
                applyMetaMarks(tr, 1, section.text.length, section.meta.get(SectionMetaOrigin) ?? [], originToMark)
                applyMetaMarks(
                    tr,
                    1,
                    section.text.length,
                    section.meta.get(SectionMetaFormatting) ?? [],
                    formattingToMark
                )
            }
            tr.setMeta('internal', true)
            return tr
        })
    }, [transformState, reload])

    const scrollCallbackRef = useRef(0)
    const handleScroll = useCallback(() => {
        clearTimeout(scrollCallbackRef.current)
        scrollCallbackRef.current = setTimeout(() => {
            unloadSections()
            reloadSections()
        }, 20)
    }, [reloadSections, unloadSections])

    useImperativeHandle(ref, () => ({
        get view() {
            return viewRef.current?.view ?? null
        },
        get root() {
            return viewRef.current?.root ?? null
        },
        get document() {
            return documentRef.current
        },
        generate,
        undo,
        redo,
        reload,
        bolden,
        italicize,
    }))

    return (
        <ProseMirror
            style={style}
            className={className}
            ref={viewRef}
            state={state}
            dispatchTransaction={dispatchTransaction}
            handleScroll={handleScroll}
        />
    )
})
