import { Transaction } from 'prosemirror-state'
import styled from 'styled-components'
import { MutableRefObject, PropsWithRef, startTransition, useEffect, useRef, useState } from 'react'

import { Document } from '../data/document/document'
import { Editor, EditorHandle } from './editor/editor'
import { documentToJsonReplacer } from '../data/document/util'

const HBox = styled.div`
    display: flex;
    flex-direction: row;
    flex-wrap: wrap;
    max-height: 100%;
`
const VBox = styled.div`
    display: flex;
    flex-direction: column;
    flex: 0;
    max-height: 100%;
`
const AppBox = styled(HBox)`
    height: 100vh;
    width: 100vw;
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, 'Open Sans',
        'Helvetica Neue', sans-serif;
    @supports (-webkit-touch-callout: none) {
        height: -webkit-fill-available;
    }
    flex-wrap: nowrap;
    overflow: hidden;
    /* background: rgb(40, 40, 40);
    color: white; */
`
const EditorContainer = styled(Editor)`
    background: rgba(101, 154, 255, 0.06);
    display: flex;
    flex: 1;
    flex-direction: column;
    overflow: auto;
    height: 100%;
`
const ButtonRow = styled.div`
    display: flex;
    flex-direction: row;
    flex-wrap: wrap;
    gap: 5px;
    padding: 8px;
`
const Button = styled.button`
    padding: 6px 16px;
    background: rgba(80, 120, 200, 0.2);
    cursor: pointer;
    user-select: none;
    &:hover {
        background: rgba(80, 120, 200, 0.3);
    }
    &[disabled] {
        background: rgba(80, 120, 200, 0.1);
    }
`

export const App = () => {
    const [transactionList, setTransactionList] = useState([] as Array<Transaction<any>>)
    const [document, setDocument] = useState<Document[]>([new Document()])
    const editorRef = useRef<EditorHandle>(null)
    const [docViewEnabled, setDocViewEnabled] = useState(true)
    const [transactionViewEnabled, setTansactionViewEnabled] = useState(false)
    return (
        <AppBox>
            <VBox style={{ minWidth: docViewEnabled ? 'calc(min(380px, 25vw))' : 'calc(min(100px, 25vw))' }}>
                {docViewEnabled ? (
                    <DocumentView document={document[0]} />
                ) : (
                    <DocumentViewContainer style={{ padding: 8 }}>disabled</DocumentViewContainer>
                )}
                <ButtonRow>
                    <Button onClick={() => setDocViewEnabled(!docViewEnabled)}>
                        {docViewEnabled ? 'Hide Document View' : 'Show...'}
                    </Button>
                </ButtonRow>
            </VBox>
            <VBox style={{ flex: 1 }}>
                <EditorContainer
                    ref={editorRef}
                    onDocumentChange={(document) => startTransition(() => setDocument([document]))}
                    onTransaction={(tr) => startTransition(() => setTransactionList((list) => [tr, ...list.slice(-2)]))}
                />
                <ButtonRow style={{ padding: 0 }}>
                    <ButtonRow>
                        <Button onClick={() => editorRef.current?.generate()}>Generate</Button>
                        <Button disabled={!document[0]?.canPopHistory()} onClick={() => editorRef.current?.undo()}>
                            Undo
                        </Button>
                        {[...(document[0]?.getDescendents()?.entries() ?? [])].map(([k]) => (
                            <Button key={k.toString()} onClick={() => editorRef.current?.redo(k)}>
                                Redo {k.toString().slice(0, 8)}...
                            </Button>
                        ))}
                    </ButtonRow>
                    <div style={{ flex: 1 }} />
                    <ButtonRow>
                        <Button onClick={() => editorRef.current?.italicize()}>Italicize</Button>
                        <Button onClick={() => editorRef.current?.bolden()}>Bolden</Button>
                        <Button onClick={() => editorRef.current?.reload()}>Reload</Button>
                    </ButtonRow>
                </ButtonRow>
            </VBox>
            <VBox style={{ width: transactionViewEnabled ? 'calc(min(380px, 25vw))' : 'calc(min(100px, 25vw))' }}>
                {transactionViewEnabled ? (
                    <TransactionList transactionList={transactionList} />
                ) : (
                    <DocumentViewContainer style={{ padding: 8 }}>disabled</DocumentViewContainer>
                )}
                <ButtonRow>
                    <Button onClick={() => setTansactionViewEnabled(!transactionViewEnabled)}>
                        {transactionViewEnabled ? 'Hide Transaction List' : 'Show...'}
                    </Button>
                </ButtonRow>
            </VBox>
        </AppBox>
    )
}

const DocumentViewContainer = styled.div`
    display: flex;
    flex-direction: column;
    flex: 1;
    overflow: auto;
    font-size: 0.675rem;
    line-height: 0.975rem;
    & > div {
        padding: 8px;
        white-space: pre;
    }
`
type DocumentViewProps = PropsWithRef<{ document?: Document }>
const DocumentView = ({ document }: DocumentViewProps) => {
    return (
        <DocumentViewContainer>
            <div>{document && document.toString()}</div>
        </DocumentViewContainer>
    )
}

const TransactionListContainer = styled.div`
    display: flex;
    flex-direction: column;
    flex: 1;
    overflow: auto;
    font-size: 0.675rem;
    line-height: 0.975rem;
    & > div {
        padding: 8px;
        white-space: pre;
    }
`
const TransactionListBackgrounds = ['rgba(250, 50, 80, 0.05)', 'rgba(30, 400, 20, 0.05)']
type TransactionListProps = PropsWithRef<{ transactionList: Array<Transaction<any>> }>
const TransactionList = ({ transactionList }: TransactionListProps) => {
    const containerRef: MutableRefObject<null | HTMLDivElement> = useRef(null)
    const colorRef = useRef(0)
    const [listRender, setListRender] = useState([<div key="-1"></div>])
    useEffect(() => {
        setListRender(
            transactionList.map((tr, idx) => (
                <div key={idx} style={{ background: TransactionListBackgrounds[(idx + colorRef.current) % 2] }}>
                    {JSON.stringify(tr, documentToJsonReplacer, 4)}
                </div>
            ))
        )
        colorRef.current += 1
        setTimeout(() => containerRef.current?.scroll({ top: -Number.MAX_SAFE_INTEGER, behavior: 'auto' }), 10)
    }, [transactionList])
    return <TransactionListContainer ref={containerRef}>{listRender}</TransactionListContainer>
}
