package routes

import (
	"encoding/json"
	"github.com/gin-gonic/gin"
	"net/http"
	"stability-image-api-middleware/generation"
	"stability-image-api-middleware/util"
	"stability-image-api-middleware/web/common"
	"strings"
)

type GenerationRequest struct {
	Model      string `json:"model,omitempty"`
	Parameters gin.H  `json:"parameters,omitempty"`
}

func Generate(c *gin.Context) {
	var request GenerationRequest

	err := c.BindJSON(&request)
	if err != nil {
		err := err.Error()
		if err == "EOF" {
			err = "No POST request body found. " +
				"Make sure it's supplied with the Content-Type header set to 'application/json'"
		}

		common.Error(c, http.StatusBadRequest, err)
		return
	}

	modelName := strings.TrimSpace(request.Model)
	if len(modelName) == 0 {
		common.Error(c, http.StatusBadRequest, "invalid model name")
		return
	}
	modelCfg, ok := generation.Models[modelName]
	if !ok {
		common.Error(c, http.StatusBadRequest, "model not found")
		return
	}

	parametersJsonString, err := json.Marshal(request.Parameters)
	if err != nil {
		util.CaptureServerError(err)
		common.Error(c, http.StatusBadRequest, "failed to marshal request parameters")
		return
	}

	req, err := http.NewRequest(http.MethodPost, modelCfg.Url, strings.NewReader(string(parametersJsonString)))
	if err != nil {
		util.CaptureServerError(err)
		common.Error(c, http.StatusBadRequest, "failed to create http request")
		return
	}
	req.Header.Set("Content-Type", "application/json")

	client := &http.Client{}
	res, err := client.Do(req)
	if err != nil {
		util.CaptureServerError(err)
		common.Error(c, http.StatusBadRequest, "failed to execute request")
		return
	}

	for key, value := range res.Header {
		c.Header(key, value[0])
	}

	c.DataFromReader(
		res.StatusCode,
		res.ContentLength,
		res.Header.Get("Content-Type"),
		res.Body,
		map[string]string{})
}
