package common

import (
	"github.com/gin-gonic/gin"
	"net/http"
	"stability-image-api-middleware/auth"
	"stability-image-api-middleware/config"
	"strings"
)

const AuthHeaderPrefix = "Bearer "
const authHeaderPrefixLength = len(AuthHeaderPrefix)

func AuthorizationMiddleware() gin.HandlerFunc {
	return func(c *gin.Context) {
		if config.Data.NoAuth {
			return
		}

		authValue := c.GetHeader("Authorization")
		if len(strings.TrimSpace(authValue)) == 0 {
			Error(c, http.StatusUnauthorized, "Authorization header not present")
			c.Abort()
			return
		}
		if !strings.HasPrefix(authValue, AuthHeaderPrefix) || len(authValue) <= authHeaderPrefixLength {
			Error(c, http.StatusBadRequest, "Malformed Authorization header")
			c.Abort()
			return
		}

		token := authValue[authHeaderPrefixLength:]
		user, err := auth.TryResolveToken(token)
		if err != nil {
			Error(c, http.StatusInternalServerError, "Failed to connect to database")
			c.Abort()
			return
		}
		if user == nil {
			Error(c, http.StatusUnauthorized, "Token not valid")
			c.Abort()
			return
		}

		c.Next()
	}
}
