package config

import (
	"os"
	"strconv"
)

type WebConfig struct {
	ListenAddress string
}

type DatabaseConfig struct {
	Backend string
	Dsn     string
}

type Config struct {
	Web      WebConfig
	Database DatabaseConfig

	NoAuth bool
	Debug  bool
}

// Simple helper function to read an environment or return a default value
func getEnv(key string, defaultVal string) string {
	if value, exists := os.LookupEnv(key); exists {
		return value
	}

	return defaultVal
}

// Helper to read an environment variable into a bool or return default value
func getEnvAsBool(name string, defaultVal bool) bool {
	valStr := getEnv(name, "")
	if val, err := strconv.ParseBool(valStr); err == nil {
		return val
	}

	return defaultVal
}

func NewConfig() *Config {
	listenPort := getEnv("PORT", "80")
	listenAddress := getEnv("LISTEN_ADDRESS", ":"+listenPort)
	return &Config{
		Web: WebConfig{
			ListenAddress: listenAddress,
		},
		Database: DatabaseConfig{
			Backend: getEnv("DB_BACKEND", "sqlite"),
			Dsn:     getEnv("DB_DSN", "middleware.db"),
		},
		NoAuth: getEnvAsBool("NO_AUTH", false),
		Debug:  getEnvAsBool("DEBUG", false),
	}
}

var Data *Config

func init() {
	Data = NewConfig()
}
