package auth

import (
	cache "github.com/go-pkgz/expirable-cache"
	"log"
	"stability-image-api-middleware/db"
	"stability-image-api-middleware/db/types"
	"time"
)

const (
	CacheMaxKeys = 1000
	CacheTTL     = 30 * time.Minute
)

var tokenCache cache.Cache

func deleteExpiredCacheEntries() {
	ticker := time.NewTicker(CacheTTL / 2)

	for {
		<-ticker.C
		tokenCache.DeleteExpired()
	}
}

func init() {
	var err error
	tokenCache, err = cache.NewCache(
		cache.LRU(),
		cache.MaxKeys(CacheMaxKeys),
		cache.TTL(CacheTTL),
	)
	if err != nil {
		log.Fatalf("failed to create API key cache: %v", err)
	}

	go deleteExpiredCacheEntries()
}

func TryResolveToken(token string) (*types.User, error) {
	// Try to lookup in cache first
	user, ok := tokenCache.Get(token)
	if ok {
		return user.(*types.User), nil
	}

	// If not found, lookup in the database
	user, err := db.UserByToken(token)
	if err != nil {
		return nil, err
	}

	// Set in cache
	tokenCache.Set(token, user, 0)

	return user.(*types.User), nil
}
