import Head from 'next/head'
import Link from 'next/link'
import { Fragment, MutableRefObject, useCallback, useEffect, useRef, useState } from 'react'
import ReactCanvasConfetti from 'react-canvas-confetti'
import { toast } from 'react-toastify'
import { useRecoilValue } from 'recoil'
import styled from 'styled-components'

import { LoginLink } from '../..'
import FileImporter, {
    FileImporterButtonType,
    FileImporterOverlayType,
} from '../../../components/controls/fileimporter'
import { FileInfo } from '../../../components/controls/fileinput'
import { LoadingSpinner } from '../../../components/loading'
import { ImportDataType } from '../../../data/story/storyconverter'
import { getUserSetting } from '../../../data/user/settings'
import { BackendURLSubmitContest } from '../../../globals/constants'
import { Session } from '../../../globals/state'
import { useAutoLogin } from '../../../hooks/useAutoLogin'
import { Dark } from '../../../styles/themes/dark'
import { InvertedButton, LightColorButton } from '../../../styles/ui/button'
import { ArrowLeftIcon, BigLightIcon, CrossMidIcon, ImageDownIcon, PartyIcon } from '../../../styles/ui/icons'
import { FlexColSpacer, FlexRow } from '../../../styles/ui/layout'
import {
    BannerMiniText,
    BannerTop,
    BannerTopContainer,
    BannerTopText,
    ContestsLayout,
    InnerContainer,
    StandbyCard,
    ToLanding,
} from '../../contest'

import Calli from '../calli.png'

export default function ContestContent(): JSX.Element {
    const [loaded, setLoaded] = useState(false)
    useAutoLogin(
        () => setLoaded(true),
        () => setLoaded(true)
    )
    return (
        <Fragment>
            <Head>
                <title>A Call For Celebration Contest - NovelAI</title>
            </Head>
            <InnerContainer>
                <FlexColSpacer min={23} max={23} />
                <FlexRow>
                    <Link href="/contest" passHref>
                        <a>
                            <FlexRow grow={false}>
                                <ArrowLeftIcon />
                                <ToLanding>Back to Details</ToLanding>
                            </FlexRow>
                        </a>
                    </Link>
                    <FlexRow grow={false}>
                        <LoginLink target="_blank" href="https://discord.com/invite/novelai">
                            Discord
                        </LoginLink>
                    </FlexRow>
                </FlexRow>
                <FlexColSpacer min={30} max={30} />
            </InnerContainer>
            <InnerContainer>
                {loaded ? (
                    <UploaderContainer>
                        <Top>
                            <BannerTop style={{ paddingBottom: 0 }}>
                                <BannerTopContainer>
                                    <BannerMiniText style={{ padding: 0, marginBottom: 0 }}>
                                        2022 NovelAI Anniversary Contest
                                    </BannerMiniText>
                                    <BannerTopText style={{ fontSize: 32 }}>
                                        Art Contest Submission
                                    </BannerTopText>
                                </BannerTopContainer>
                            </BannerTop>
                        </Top>
                        <StandbyCard />
                        {/* <Info />
                        <Bottom>
                            <Uploader allowedExtensions={['png', 'jpg', 'jpeg', 'gif', 'mp4']} />
                        </Bottom> */}
                    </UploaderContainer>
                ) : (
                    <LoadingSpinner visible />
                )}
            </InnerContainer>
        </Fragment>
    )
}
ContestContent.Layout = ContestsLayout

function Info() {
    return (
        <InfoBox>
            <strong>Art must be:</strong>
            <br />
            🞄 Created by the submitter
            <br />
            🞄 Related to the Theme
            <br />
            🞄 A visual media at a reasonable size (max file size 50MB)
            <br />
            <br />
            <strong>Art may not include:</strong>
            <br />
            🞄 Pre-existing IPs of any kind
            <br />
            🞄 Overtly sexually explicit content.
        </InfoBox>
    )
}

const Top = styled.div`
    padding: 20px;
    padding-top: 80px;
    display: flex;
    flex-direction: column;
    justify-content: flex-end;
    position: relative;
    background-color: #13152c;
    &::before {
        content: '';
        pointer-events: none;
        width: 100%;
        height: 100%;
        position: absolute;
        top: 0;
        left: 0;
        background: url(${Calli.src});
        background-position: top right;
        background-repeat: no-repeat;
        background-size: contain;
        @media screen and (max-width: 650px) {
            opacity: 0.5;
        }
        @media screen and (max-width: 500px) {
            background-position: center bottom;
            opacity: 0.3;
        }
    }
`
const Bottom = styled.div`
    padding: 20px;
    background: #191b31;
`
const InfoBox = styled.div`
    padding: 20px;
    border-bottom: 2px solid #22253f;
    background: #191b31;
`
const UploaderContainer = styled.div`
    width: 100%;
    max-width: min(600px, 100vw);
    border: 2px solid #22253f;
`
interface UploaderProps {
    children?: JSX.Element | JSX.Element[] | string
    allowedExtensions: string[]
}
function Uploader({ children, allowedExtensions }: UploaderProps) {
    const [file, setFile] = useState(null as null | FileInfo)
    const onFileImport = async (file: FileInfo) => {
        const extension = file.name.slice(file.name.lastIndexOf('.') + 1)
        if (!allowedExtensions.includes(extension)) {
            toast('File type not supported.')
            return true
        }
        if (file.size > 50000000) {
            toast('Maximum allowed file size is 50MB.')
            return true
        }
        setFile(file)
        return true
    }
    const session = useRecoilValue(Session)
    const [disabled, setDisabled] = useState(true)
    const [name, setName] = useState(getUserSetting(session.settings, 'penName') || '')
    const [email, setEmail] = useState('')
    const [socials, setSocials] = useState('')
    const [mediums, setMediums] = useState('')
    const [error, setError] = useState('')
    const [success, setSuccess] = useState('')
    const [loading, setLoading] = useState(true)
    useEffect(() => {
        if (!session.authenticated || session.noAccount) {
            setLoading(false)
            return
        }
        fetch(BackendURLSubmitContest + '/anniversary2022-art', {
            mode: 'cors',
            cache: 'no-store',
            headers: {
                'Content-Type': 'application/json',
                Authorization: 'Bearer ' + session.auth_token,
            },
            method: 'GET',
        })
            .then((r) => r.text())
            .then((text) => {
                if (!text) return
                const json = JSON.parse(text)
                if (json.statusCode && json.statusCode !== 200) throw json.message
                setName(json.authorName)
                setEmail(json.authorEmail)
                setSocials(json.socials)
                setMediums(json.mediums)
                setFile({
                    buff: new ArrayBuffer(0),
                    name: json.dataName,
                    size: 0,
                    text: '',
                    type: ImportDataType.unknown,
                })
            })
            .catch((error_) => setError(`${error_}`))
            .finally(() => setLoading(false))
    }, [session.auth_token, session.authenticated, session.noAccount])
    const confettiRef: MutableRefObject<any> = useRef(null)
    const submit = useCallback(() => {
        if (disabled || !file || !name || !email || file.size <= 2) return
        const doSubmit = async () => {
            setLoading(true)
            setError('')
            setSuccess('')
            try {
                const request = await fetch(BackendURLSubmitContest, {
                    mode: 'cors',
                    cache: 'no-store',
                    headers: {
                        'Content-Type': 'application/json',
                        Authorization: 'Bearer ' + session.auth_token,
                    },
                    method: 'POST',
                    body: JSON.stringify({
                        authorName: name,
                        authorEmail: email,
                        event: 'anniversary2022-art',
                        dataName: file.name,
                        socials,
                        mediums,
                        data: Buffer.from(file.buff).toString('base64'),
                    }),
                })
                const json = await request.json()
                if (json.statusCode && json.statusCode !== 200) throw json.message
                setSuccess(
                    'Thank you for your submission! You can update your submission until the submission deadline.'
                )
                setFile({
                    ...file,
                    size: 0,
                })
                setTimeout(
                    () =>
                        confettiRef.current?.confetti({
                            spread: 38,
                            startVelocity: 14,
                            origin: { y: 0.5 },
                            particleCount: Math.floor(45),
                            colors: [Dark.colors.textHeadings],
                            scalar: 0.75,
                        }),
                    10
                )
            } catch (error_: any) {
                setError(`${error_}`)
            } finally {
                setLoading(false)
            }
        }
        doSubmit()
    }, [disabled, email, file, mediums, name, session.auth_token, socials])
    const clickRef = useRef(() => false)
    useEffect(() => {
        setDisabled(!(name.length > 0 && email.length > 3 && email.includes('@') && (file?.size ?? 0) > 2))
    }, [email, file, name.length])
    return !session.authenticated || session.noAccount ? (
        <Error>
            <BigLightIcon />
            <span>
                You need to be <Link href="/login">registered and logged in</Link> to participate.
            </span>
        </Error>
    ) : (
        <Fragment>
            <h4>Author Name</h4>
            <input
                type="text"
                name="Author Name"
                value={name}
                onChange={(e) => setName(e.target.value)}
                disabled={loading}
            />
            <FlexColSpacer min={20} max={20} />
            <h4>Contact Email</h4>
            <input
                type="email"
                name="Contact Email"
                value={email}
                onChange={(e) => setEmail(e.target.value)}
                disabled={loading}
            />
            <FlexColSpacer min={20} max={20} />
            <h4>Socials</h4>
            <textarea
                name="Socials"
                value={socials}
                onChange={(e) => setSocials(e.target.value)}
                disabled={loading}
            />
            <FlexColSpacer min={20} max={20} />
            <h4>What medium(s) did you use to create your submission?</h4>
            <textarea
                name="Mediums"
                value={mediums}
                onChange={(e) => setMediums(e.target.value)}
                disabled={loading}
            />
            <FlexColSpacer min={20} max={20} />
            <FileImporter
                overlay={FileImporterOverlayType.Fixed}
                button={FileImporterButtonType.Direct}
                buttonClickRef={clickRef}
                onImportFile={onFileImport}
                allowedFileTypes={[ImportDataType.unknown]}
            >
                <CC>
                    <CC1>
                        <div>Your Art Submission</div>
                        <div>Supported Types: .png .jpg .gif .mp4</div>
                    </CC1>
                    <LightColorButton
                        style={{ flex: '1 1 auto', justifyContent: 'center' }}
                        disabled={loading}
                        onClick={() => clickRef.current()}
                    >
                        Select File
                    </LightColorButton>
                </CC>
            </FileImporter>
            <FlexColSpacer min={20} max={20} />
            {file && <FilePreview file={file} remove={() => setFile(null)} />}
            <FlexColSpacer min={20} max={20} />
            <InvertedButton style={{ width: '100%' }} disabled={disabled || loading} onClick={submit}>
                Submit
            </InvertedButton>
            {error && (
                <Error>
                    <CrossMidIcon style={{ height: 20, width: 20 }} />
                    <span>{error}</span>
                </Error>
            )}
            {success && (
                <Success>
                    <Confetti ref={confettiRef} />
                    <PartyIcon style={{ height: 40, width: 40 }} /> <span>{success}</span>
                </Success>
            )}
            {children}
        </Fragment>
    )
}
export const Confetti = styled(ReactCanvasConfetti)`
    position: absolute;
    max-width: unset;
    width: 150px;
    height: 300px;
    bottom: -120px;
    left: -60px;
    z-index: 1000;
    &,
    & * {
        pointer-events: none !important;
    }
`
const Error = styled.div`
    color: ${Dark.colors.warning};
    padding: 10px 0;
    display: flex;
    flex-direction: row;
    align-items: center;
    gap: 20px;
    position: relative;
`
const Success = styled(Error)`
    color: ${Dark.colors.textHeadingsOptions[2]};
`
const CC = styled.div`
    display: flex;
    flex-direction: row;
    width: 100%;
`
const CC1 = styled.div`
    flex: 10 1 auto;
    & > div:nth-child(1) {
        font-size: 16;
        font-weight: 600;
        font-family: ${() => Dark.fonts.headings};
    }
    & > div:nth-child(2) {
        opacity: 0.5;
    }
`

const FileOuterContainer = styled.div`
    display: flex;
    flex-direction: column;
    gap: 10px;
    padding: 15px;
    background: ${() => Dark.colors.bg0};
`
const FileInfoContainer = styled.div`
    display: flex;
    flex-direction: row;
    flex-wrap: wrap;
    gap: 10px;
    position: relative;
    overflow: hidden;
    padding: 10px;
    background: ${() => Dark.colors.bg2};
`
const FileInfoPreview = styled.div`
    height: 100px;
    min-width: 100px;
    max-width: 100%;
    border: 1px solid white;
    position: relative;
    pointer-events: none;
    display: flex;
    justify-content: center;
    align-items: center;
    flex: 1;
`
const FileInfoName = styled.div`
    padding: 10px;
    display: flex;
    flex-direction: column;
    &::before {
        content: 'Name';
        opacity: 0.5;
        font-size: 0.675;
    }
    flex: 10;
`
const Delete = styled.div`
    padding: 5px;
    opacity: 0.8;
    display: flex;
    align-items: center;
    cursor: pointer;
    &:hover {
        opacity: 1;
    }
`
interface FilePreviewProps {
    file: FileInfo
    remove: () => void
}
function FilePreview({ file, remove }: FilePreviewProps) {
    const extension = file.name.slice(file.name.lastIndexOf('.') + 1)
    return (
        <FileOuterContainer>
            <FileInfoContainer>
                <FileInfoPreview>
                    {['gif', 'png', 'jpg', 'jpeg'].includes(extension) && file.buff.byteLength > 5 ? (
                        <img
                            src={`data:image/${extension};base64,${Buffer.from(file.buff).toString(
                                'base64'
                            )}`}
                            width="100%"
                            height="100%"
                            style={{ maxWidth: '100%', objectFit: 'cover' }}
                            alt="Preview"
                        />
                    ) : (
                        <ImageDownIcon />
                    )}
                </FileInfoPreview>
                <FileInfoName>{file.name}</FileInfoName>
                {/* <Delete onClick={() => remove()}>
                    <CrossMidIcon />
                </Delete> */}
            </FileInfoContainer>
        </FileOuterContainer>
    )
}
