
const dir = './groups'
const adventureDir = './textAdventure'
const menagerieDir = './menagerie'
const anniversaryDir = './anniversary'

const fs = require('fs')
const pc = require('picocolors')

const settingsOverrides = {
    max_length: 40,
    min_length: 1,
}
const settingsOverridesIgnores = {
    max_length: (v) => v <= 30, // ignore purposely low lengths
    min_length: () => false,
}

const alphaSettingsOne = {
    temperature: 0.8,
    top_k: 50,
    top_p: 0.9,
    tail_free_sampling: 1,
    repetition_penalty: 2.0,
    repetition_penalty_range: 512,
    repetition_penalty_slope: 3.33,
}

const alphaSettingsTwo = {
    temperature: 0.85,
    top_k: 60,
    top_p: 0.9,
    tail_free_sampling: 1,
    repetition_penalty: 2.8,
    repetition_penalty_range: 512,
    repetition_penalty_slope: 3.33,
}

const betaSettings = {
    temperature: 0.55,
    top_k: 140,
    top_p: 0.9,
    tail_free_sampling: 1,
    repetition_penalty: 3.5,
    repetition_penalty_range: 1024,
    repetition_penalty_slope: 6.57,
}

const storywriter = {
    temperature: 0.72,
    top_k: 0,
    top_p: 0.725,
    tail_free_sampling: 1,
    repetition_penalty: 2.75,
    // ignore repetition_penalty_range
    repetition_penalty_slope: 0.18,
}

const adjustedStorywriter = {
    temperature: 0.72,
    top_k: 0,
    top_p: 0.725,
    tail_free_sampling: 1,
    repetition_penalty: 2.75,
    // ignore repetition_penalty_range
    repetition_penalty_slope: 0.18,
}

const migratableSettings = [
    { name: 'alpha1', template: alphaSettingsOne, ignore: 2 },
    { name: 'alpha2', template: alphaSettingsTwo, ignore: 2 },
    { name: 'beta', template: betaSettings, ignore: 2 },
    { name: 'storywriter', template: storywriter, ignore: 0 },
    { name: 'adjustedStorywriter', template: adjustedStorywriter, ignore: 0 },
]

const genesis = {
    temperature: 0.63,
    top_k: 0,
    top_p: 0.975,
    tail_free_sampling: 0.975,
    repetition_penalty: 2.975,
    repetition_penalty_range: 2048,
    repetition_penalty_slope: 0.09,
    repetition_penalty_frequency: 0,
    repetition_penalty_presence: 0,
    order: [
        {
            id: 'top_p',
            enabled: true,
        },
        {
            id: 'top_k',
            enabled: true,
        },
        {
            id: 'tfs',
            enabled: true,
        },
        {
            id: 'temperature',
            enabled: true,
        },
        {
            id: 'top_a',
            enabled: false,
        },
        {
            id: 'typical_p',
            enabled: false,
        },
    ],
}

function isSameSettings(settings, template) {
    let same = true
    let diff = []
    for (const key of Object.keys(template)) {
        if (settings[key] !== template[key]) {
            same = false
            diff = [...diff, key]
        }
    }
    return { same, diff }
}

function migrateSettings(scenario) {
    let diff = []
    if (scenario.settings.model === 'euterpe-v2') {
        // don't migrate euterpe scenarios
        return { migrated: false, diff: 0, adjusted: [], alreadyGood: true }
    }
    if (scenario.settings.prefix && scenario.settings.prefix !== 'theme_textadventure') {
        // can't migrate custom module scenarios
        return { migrated: false, diff: 0, adjusted: [], moduleIncompatible: true }
    }
    // if settings are equal to existing preset, migrate
    for (const { name, template } of migratableSettings) {
        const same = isSameSettings(scenario.settings.parameters, template)
        if (same.same) {
            scenario.settings.parameters = { ...scenario.settings.parameters, ...genesis, ...settingsOverrides }
            return { migrated: true, from: name, ignored: [] }
        } else {
            diff = [...diff, { name, diff: same.diff }]
        }
    }
    // if max difference to an existing preset is under difference limit, ignore and migrate
    const maxDiff = findMaxDiff(diff)
    if (maxDiff.diff.length <= migratableSettings.find(m => m.name === maxDiff.name).ignore) {
        scenario.settings.parameters = { ...scenario.settings.parameters, ...genesis, ...settingsOverrides }
        return { migrated: true, from: maxDiff.name, ignored: maxDiff.diff }
    }
    let adjusted = []
    // if diff is storywriter, adjust repetition_penalty_range
    if (maxDiff.name === 'storywriter' && scenario.settings.parameters.repetition_penalty_range !== genesis.repetition_penalty_range) {
        scenario.settings.parameters.repetition_penalty_range = genesis.repetition_penalty_range
        adjusted = [...adjusted, 'repetition_penalty_range']
    }
    if (diff.length === 0 || Object.entries(settingsOverrides).length > 0) {
        // set scenario model to euterpe for updated settings
        scenario.settings.model = 'euterpe-v2'
    }
    // override defaults
    for (const [key, value] of Object.entries(settingsOverrides)) {
        if (scenario.settings.parameters[key] !== value && !settingsOverridesIgnores[key](scenario.settings.parameters[key])) {
            scenario.settings.parameters[key] = value
            adjusted = [...adjusted, key]
        }
    }
    return { migrated: false, diff, maxDiff, adjusted }
}

function findMaxDiff(diffs) {
    return diffs.reduce((a, b) => a.diff.length < b.diff.length ? a : b)
}

async function createDefaults(dir, filename) {
    console.log(`Creating bundled file for dir '${dir}'`)
    const scenarioGroups = []
    const migrations = []
    const outliers = []
    const authors = fs.readdirSync(dir)
    for (const author of authors) {
        const scenarios = fs.readdirSync(dir + '/' + author)
        for (const scenario of scenarios) {
            const scenarioGroup = {}
            scenarioGroup.name = scenario.replace('[col]', ':')
            scenarioGroup.name = scenarioGroup.name.replace('[p]', '.')
            scenarioGroup.names = []
            scenarioGroup.scenarios = []
            const subScenarios = fs.readdirSync(dir + '/' + author + '/' + scenario)
            for (const subScenario of subScenarios) {
                scenarioGroup.names.push(subScenario)
                const scenarioFiles = fs.readdirSync(dir + '/' + author + '/' + scenario + '/' + subScenario)
                for (const file of scenarioFiles) {
                    const filecontents = fs.readFileSync(dir + '/' + author + '/' + scenario + '/' + subScenario + '/' + file, 'utf-8')

                    const scenarioObject = JSON.parse(filecontents)
                    scenarioObject.author = author

                    const migrated = migrateSettings(scenarioObject)
                    if (migrated.migrated) {
                        console.log(`Migrated settings for ${pc.green(scenarioObject.title)} from ${pc.yellow(`${migrated.from}`)}${migrated.ignored.length ? pc.gray(` (ignored ${migrated.ignored})`) : ''}`)
                        migrations.push({ scenario: scenarioObject })
                    } else {
                        outliers.push({ scenario: scenarioObject, diff: migrated.diff, maxDiff: migrated.maxDiff, adjusted: migrated.adjusted, alreadyGood: migrated.alreadyGood, moduleIncompatible: migrated.moduleIncompatible })
                    }
                    scenarioGroup.scenarios.push(scenarioObject)
                }
            }
            if (scenarioGroup.scenarios.length > 0) { scenarioGroups.push(scenarioGroup) } else {
                console.warn(pc.red(`Skipped scenario group ${scenarioGroup.name} without scenarios`))
            }
        }
    }
    if (outliers) console.log(`Outliers where settings could not be migrated: ${pc.red(outliers.length)} (out of ${migrations.length + outliers.length})`)
    for (const outlier of outliers) {
        if (outlier.alreadyGood) {
            console.log(` - ${pc.blue(outlier.scenario.title)} (${pc.bgGreen('already good!')})`)
        } else if (outlier.moduleIncompatible) {
            console.log(` - ${pc.blue(outlier.scenario.title)} (${pc.bgRed('modules incompatible!')})`)
        } else {
            console.log(` - ${pc.blue(outlier.scenario.title)} (most similar to ${pc.yellow(outlier.maxDiff.name)} with diff in ${pc.gray(outlier.maxDiff.diff)})${outlier.adjusted.length ? pc.italic(` (adjusted ${outlier.adjusted})`) : ''}`)
        }
    }
    fs.writeFileSync(filename, JSON.stringify(scenarioGroups))

    console.log(`${pc.inverse(`Output bundled file to ${filename}`)}`)
}

async function createMenagerieDefaults(dir, filename) {
    console.log(`Creating bundled file for dir '${dir}'`)
    const scenarioGroups = []
    const migrations = []
    const outliers = []
    const scenarios = fs.readdirSync(dir)
    for (const scenarioFile of scenarios) {
        const scenarioGroup = {}
        scenarioGroup.names = []
        scenarioGroup.scenarios = []
        scenarioGroup.id = 'hw21' + scenarioFile.split('.')[0].slice(16)

        const filecontents = fs.readFileSync(dir + '/' + scenarioFile, 'utf-8')

        const scenarioObject = JSON.parse(filecontents)
        scenarioObject.tags = [...scenarioObject.tags, 'halloween', 'menagerie']
        const name = scenarioFile.split('.')[0].slice(16)
        if (name === 'hYU-rGuONKXaWjTiCaF9F') {
            scenarioObject.tags = [...scenarioObject.tags, '1st place']
        } else if (name === 'xnckWDeJ-D1OHSGEq3_Ec') {
            scenarioObject.tags = [...scenarioObject.tags, '2nd place']
        } else if (name === 'DUE87-X7Ap6fjdJMB1jf4') {
            scenarioObject.tags = [...scenarioObject.tags, '3rd place']
        } else if (name === 'H7EGmRdIt0oF95lKu7yAk') {
            scenarioObject.tags = [...scenarioObject.tags, '3rd place']
        } else if (name === 'wPzgFVU2eee5_0gdkR-YO') {
            scenarioObject.tags = [...scenarioObject.tags, 'staff pick']
        } else {
            continue
        }
        scenarioGroup.name = scenarioObject.title
        scenarioGroup.names.push('Text Adventure')
        scenarioGroup.scenarios.push(scenarioObject)

        const migrated = migrateSettings(scenarioObject)
        if (migrated.migrated) {
            console.log(`Migrated settings for ${pc.green(scenarioObject.title)} from ${pc.yellow(`${migrated.from}`)}${migrated.ignored.length ? pc.gray(` (ignored ${migrated.ignored})`) : ''}`)
            migrations.push({ scenario: scenarioObject })
        } else {
            outliers.push({ scenario: scenarioObject, diff: migrated.diff, maxDiff: migrated.maxDiff, adjusted: migrated.adjusted })
        }

        scenarioGroups.push(scenarioGroup)
    }
    if (outliers) console.log(`Outliers where settings could not be migrated: ${pc.red(outliers.length)} (out of ${migrations.length + outliers.length})`)
    for (const outlier of outliers) {
        console.log(` - ${pc.blue(outlier.scenario.title)} (most similar to ${pc.yellow(outlier.maxDiff.name)} with diff in ${pc.gray(outlier.maxDiff.diff)})${outlier.adjusted.length ? pc.italic(` (adjusted ${outlier.adjusted})`) : ''}`)
    }
    fs.writeFileSync(filename, JSON.stringify(scenarioGroups))

    console.log(`${pc.inverse(`Output bundled file to ${filename}`)}`)
}

async function createAnniversaryDefaults(dir, filename) {
    console.log(`Creating bundled file for dir '${dir}'`)
    const scenarioGroups = []
    const migrations = []
    const outliers = []
    const scenarios = fs.readdirSync(dir)
    for (const scenarioFile of scenarios) {
        const scenarioGroup = {}
        scenarioGroup.names = []
        scenarioGroup.scenarios = []
        scenarioGroup.id = '' + scenarioFile.split('.')[0].slice(16)

        const filecontents = fs.readFileSync(dir + '/' + scenarioFile, 'utf-8')

        const scenarioObject = JSON.parse(filecontents)
        if (!scenarioObject.tags.includes('anniversary contest')) scenarioObject.tags.push('anniversary contest')
        if (!scenarioObject.tags.includes('celebration')) scenarioObject.tags.push('celebration')

        const name = scenarioFile.split('.')[0].slice(16)
        if (name === '0ykfb-Nx1JAgKqRnoZ36T') {
            scenarioObject.tags.push('1st place')
            scenarioGroup.names.push('1st')
        } else if (name === 'CIVFGS0ruCdlNopegTWOt') {
            scenarioObject.tags.push('2nd place')
            scenarioGroup.names.push('1st')
        } else if (name === 'QLFG2745njIyKA9C1wvBt') {
            scenarioObject.tags.push('3rd place')
            scenarioGroup.names.push('1st')
        } else {
            continue
        }

        scenarioGroup.name = scenarioObject.title
        scenarioGroup.scenarios.push(scenarioObject)

        scenarioGroups.push(scenarioGroup)
    }

    fs.writeFileSync(filename, JSON.stringify(scenarioGroups))

    console.log(`${pc.inverse(`Output bundled file to ${filename}`)}`)
}

createDefaults(dir, './defaultScenarios.json')
createDefaults(adventureDir, './defaultTextAdventureScenarios.json')
createMenagerieDefaults(menagerieDir, './defaultMenagerieScenarios.json')
createAnniversaryDefaults(anniversaryDir, './defaultAnniversaryScenarios.json')
