﻿import React, { useCallback, useMemo } from 'react'
import LinearGradient from 'react-native-linear-gradient'
import { useRecoilValue } from 'recoil'
import { normalizeModel, TextGenerationModel } from '../../shared/data/request/model'
import { PrefixOptions, CustomPrefixImage } from '../../shared/data/story/defaultprefixes'
import { AIModule } from '../../shared/data/story/storysettings'
import { GlobalUserContext } from '../../shared/globals/globals'
import { CustomModules, SelectedStoryId, SiteTheme } from '../../shared/globals/state'
import { modelsCompatible, prefixModel } from '../../shared/util/models'
import { isModuleImageValid } from '../../shared/util/util'
import { SelectorSection } from '../components/modals/selectormodal'
import { hexOpacity } from '../util/colors'
import { ImageSourcePropType, View } from 'react-native'
import styled from 'styled-components/native'
import { BodyMedium600M } from '../styles/fonts'
import { PlatformImageData } from '../../shared/compatibility/platformtypes'

export const themes = [...PrefixOptions.keys()].filter((prefix) =>
    PrefixOptions.get(prefix)?.label.startsWith('Theme:')
)
export const styles = [...PrefixOptions.keys()].filter((prefix) =>
    PrefixOptions.get(prefix)?.label.startsWith('Style:')
)
export const inspirations = [...PrefixOptions.keys()].filter((prefix) =>
    PrefixOptions.get(prefix)?.label.startsWith('Inspiration:')
)
export const specials = [...PrefixOptions.keys()].filter((prefix) =>
    PrefixOptions.get(prefix)?.label.startsWith('Special:')
)
export const generals = [...PrefixOptions.keys()].filter((prefix) =>
    PrefixOptions.get(prefix)?.label.startsWith('General:')
)

export function filterModules(model: TextGenerationModel | undefined): (_: AIModule) => boolean {
    return (module: AIModule) => {
        if (!model) return false
        return modelsCompatible(normalizeModel(prefixModel(module.id)), normalizeModel(model))
    }
}

export function ModuleOptionVisual(props: {
    image: ImageSourcePropType
    name: string | undefined
    selected: boolean
}): JSX.Element {
    const siteTheme = useRecoilValue(SiteTheme)
    const backgroundColor = props.selected ? siteTheme.colors.bg1 : siteTheme.colors.bg2

    //need to change the image format. Currently, because StaticImageData has no src parameter, the prefixes file is wrong
    //also, it is necessary to prepare for imported prefix images, and make sure they can be loaded fine
    //also, the gradient here will not match the background of the selected label button.
    //when used there, it should be considered not selected, but how? It might be necessary to use a different component from the label in that case
    //which means it might be necessary to modify or split the selector modal file, which is annoying
    return (
        <View style={{ flexDirection: 'row' }}>
            <ModuleImage source={props.image}>
                <LinearGradient
                    start={{ x: 0, y: 0 }}
                    end={{ x: 1, y: 0 }}
                    colors={[hexOpacity(backgroundColor, 0), hexOpacity(backgroundColor, 1.0)]}
                    style={{ position: 'absolute', width: '100%', height: '100%' }}
                />
            </ModuleImage>
            <ModuleText>{props.name ?? 'Unknown Module'}</ModuleText>
        </View>
    )
}

export const ModuleImage = styled.ImageBackground`
    width: 110px;
    height: 50px;
`

export const ModuleText = styled.Text`
    ${BodyMedium600M}
    align-self: center;
    padding-left: 10px;
`

export function useModuleOptions(
    selectedModule: string,
    model: TextGenerationModel | undefined,
    checkStory: boolean = false
): SelectorSection[] {
    const customModules = useRecoilValue(CustomModules)
    const selectedStory = useRecoilValue(SelectedStoryId)
    const currentStoryContent = GlobalUserContext.storyContentCache.get(selectedStory)
    const hasStorySpecificPrefix =
        currentStoryContent &&
        currentStoryContent.settings &&
        currentStoryContent.settings.aiModule &&
        !customModules.some((e) => e.id === currentStoryContent?.settings?.aiModule?.id)

    const mapToOptions = useCallback(
        (key: string) => ({
            value: key,
            description: PrefixOptions.get(key)?.label ?? key,
            rawLabel: PrefixOptions.get(key)?.label ?? key,
            label: (
                <ModuleOptionVisual
                    image={PrefixOptions.get(key)?.image ?? CustomPrefixImage}
                    name={(PrefixOptions.get(key)?.label ?? key).replace(
                        /^((theme|style|inspiration|general|special):? *)/i,
                        ''
                    )}
                    selected={key === selectedModule}
                />
            ),
        }),
        [selectedModule]
    )

    const storyOptions = useMemo(() => {
        return checkStory &&
            hasStorySpecificPrefix &&
            modelsCompatible(
                currentStoryContent?.settings.model,
                prefixModel(currentStoryContent?.settings?.aiModule?.id ?? '')
            )
            ? [
                  {
                      title: 'Story',
                      data: [
                          {
                              value: currentStoryContent?.settings?.aiModule?.id,
                              description: currentStoryContent?.settings?.aiModule?.description,
                              rawLabel: currentStoryContent?.settings?.aiModule?.name ?? 'Unknown Module',
                              label: (
                                  <ModuleOptionVisual
                                      image={
                                          (currentStoryContent?.settings?.aiModule
                                              ?.image as PlatformImageData) ?? CustomPrefixImage
                                      }
                                      name={currentStoryContent?.settings?.aiModule?.name}
                                      selected={
                                          currentStoryContent?.settings?.aiModule?.id === selectedModule
                                      }
                                  />
                              ),
                          },
                      ],
                  },
              ]
            : []
    }, [
        checkStory,
        currentStoryContent?.settings?.aiModule?.description,
        currentStoryContent?.settings?.aiModule?.id,
        currentStoryContent?.settings?.aiModule?.image,
        currentStoryContent?.settings?.aiModule?.name,
        currentStoryContent?.settings.model,
        hasStorySpecificPrefix,
        selectedModule,
    ])

    const importedOptions = useMemo(
        () => [
            {
                title: 'Imported',
                data: customModules.filter(filterModules(model)).map((module) => ({
                    value: module.id,
                    description: module.description,
                    rawLabel: module.name,
                    label: (
                        <ModuleOptionVisual
                            image={(module.image as PlatformImageData) ?? CustomPrefixImage}
                            name={module.name}
                            selected={module.id === selectedModule}
                        />
                    ),
                })),
            },
        ],
        [customModules, model, selectedModule]
    )

    const defaultOptions = useMemo(
        () => [
            {
                title: 'Specials',
                data: specials.map((element) => mapToOptions(element)),
            },
            {
                title: 'General',
                data: generals.map((element) => mapToOptions(element)),
            },
            {
                title: 'Styles',
                data: styles.map((element) => mapToOptions(element)),
            },
            {
                title: 'Themes',
                data: themes.map((element) => mapToOptions(element)),
            },
            {
                title: 'Inspirations',
                data: inspirations.map((element) => mapToOptions(element)),
            },
        ],
        [mapToOptions]
    )

    const vanillaOptions = useMemo(
        () => [
            {
                title: 'Vanilla',
                data: [mapToOptions([...PrefixOptions.keys()][0])],
            },
        ],
        [mapToOptions]
    )

    const prefixOptions = useMemo(() => {
        const options: any = []
        options.push(...vanillaOptions, ...storyOptions, ...importedOptions, ...defaultOptions)
        return options
    }, [defaultOptions, importedOptions, storyOptions, vanillaOptions])

    return prefixOptions
}
